<?php

namespace MetaFox\Platform\Repositories;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use MetaFox\Core\Support\Facades\Language;
use MetaFox\Localize\Support\Browse\Scopes\Phrase\TranslatableTextSearchScope;
use MetaFox\Platform\Contracts\User;
use MetaFox\Platform\Facades\ResourceGate;
use MetaFox\Platform\MetaFoxConstant;
use MetaFox\Platform\Repositories\Contracts\CategoryRepositoryInterface;

/**
 * Trait HasApprove.
 * @codeCoverageIgnore
 * @SuppressWarnings(PHPMD.UnusedFormalParameter)
 */
abstract class AbstractCategoryRepository extends AbstractRepository implements CategoryRepositoryInterface
{
    public function createCategory(User $context, array $attributes): Model
    {
        $attributes['is_active'] = Arr::get($attributes, 'is_active', 0);

        $parentId = Arr::get($attributes, 'parent_id');

        $attributes['level'] = 1;

        if ($parentId) {
            $parent = $this->find($parentId);

            $attributes['level'] += $parent->level;
        }

        if ($attributes['level'] > MetaFoxConstant::MAX_CATEGORY_LEVEL) {
            abort(403, json_encode([
                'title'   => __p('core::phrase.content_is_not_available'),
                'message' => __p('core::phrase.it_is_not_allowed_to_move_this_category_and_subcategories_to_the_selected_category', [
                    'value' => MetaFoxConstant::MAX_CATEGORY_LEVEL,
                ]),
            ]));
        }

        $attributes['ordering'] = $this->getNextOrdering($attributes['level']);

        $category = $this->getModel()->newQuery()->create($attributes);

        $category->refresh();

        $this->createCategoryRelationFor(category: $category);
        $this->clearCache();

        return $category;
    }

    protected function getNextOrdering(int $level): int
    {
        $currentCategory = $this->getModel()->newQuery()
            ->where([
                'level' => $level,
            ])
            ->orderByDesc('ordering')
            ->first();

        if (null === $currentCategory) {
            return 0;
        }

        return (int) $currentCategory->ordering + 1;
    }

    public function updateCategory(User $context, int $id, array $attributes): Model
    {
        $category    = $this->find($id);
        $oldParentId = $category->parent_id;
        $newParentId = Arr::get($attributes, 'parent_id');

        if ($newParentId !== null && $newParentId !== $oldParentId) {
            $newCategory         = $this->find($newParentId);
            $attributes['level'] = $newCategory->level + 1;

            $this->changeParentCategory($category, $newParentId);
        }

        if ($newParentId == null) {
            $this->decrementTotalItemCategories($category?->parentCategory, $category->total_item);
            $attributes['level'] = 1;
        }

        $category->fill($attributes)->save();
        $category->refresh();
        $this->clearCache();

        return $category;
    }

    public function deleteOrMoveToNewCategory(Model $category, int $newCategoryId): bool
    {
        if ($newCategoryId > 0) {
            $this->moveToNewCategory($category, $newCategoryId, true);

            $this->clearCache();

            return (bool) $category->forceDelete();
        }

        $this->deleteAllBelongTo($category);
        $this->deleteCategoryRelations(category: $category);
        $this->clearCache();

        return (bool) $category->forceDelete();
    }

    public function getCategoriesForForm(): array
    {
        return $this->getModel()->newQuery()
            ->select('id as value', 'name as label', 'parent_id', 'is_active', 'level', 'ordering')
            ->where('is_active', MetaFoxConstant::IS_ACTIVE)
            ->orderBy('ordering')
            ->get()
            ->toArray();
    }

    public function getCategoriesForStoreForm(?Model $category): array
    {
        $query = $this->getModel()->newQuery()
            ->where('is_active', MetaFoxConstant::IS_ACTIVE);
        $query->where('level', '<', MetaFoxConstant::MAX_CATEGORY_LEVEL);

        if (null !== $category) {
            $query->where('id', '<>', $category->entityId());
            $query->where('level', '<=', $category->level);

            if ($category->subCategories()->exists()) {
                $query->where('level', '<', $category->level);
            }

            if ($category->subCategories()->exists() && !$category->parentCategory()->exists()) {
                return [];
            }
        }

        return $query->select('id as value', 'name as label', 'parent_id', 'is_active', 'level')
            ->orderBy('ordering')
            ->get()
            ->toArray();
    }

    public function getCategoryForFilter(): Collection
    {
        return $this->getModel()->newQuery()
            ->with([
                'subCategories' => function (HasMany $q) {
                    $q->where('is_active', MetaFoxConstant::IS_ACTIVE)
                        ->orderBy('ordering');
                },
            ])
            ->whereNull('parent_id')
            ->where('is_active', MetaFoxConstant::IS_ACTIVE)
            ->orderBy('ordering')
            ->get()
            ->collect();
    }

    public function getStructure(User $context, array $attributes): array
    {
        if (empty(Arr::except($attributes, ['limit']))) {
            return localCacheStore()
                ->rememberForever(
                    $this->getStructureCacheKey(),
                    fn () => ResourceGate::items($this->getAllCategories($context, []))
                );
        }

        return ResourceGate::items($this->getAllCategories($context, $attributes));
    }

    public function getAllCategories(User $context, array $attributes): Collection
    {
        $idsInactive = $this->getCategoryInactive();
        $search      = Arr::get($attributes, 'q');
        $level       = Arr::get($attributes, 'level', 1);
        $table       = $this->getModel()->getTable();

        $query = $this->getModel()->newQuery()
            ->where('is_active', MetaFoxConstant::IS_ACTIVE)
            ->where(function (Builder $builder) use ($idsInactive, $table) {
                $builder->whereNotIn("$table.id", array_unique($idsInactive));
            });

        if ($search !== null) {
            $defaultLocale = Language::getDefaultLocaleId();
            $searchScope   = new TranslatableTextSearchScope($search, ['ps.text']);
            $searchScope->setLocale($defaultLocale);

            return $query->addScope($searchScope)->get(["$table.*"])->collect();
        }

        if (array_key_exists('id', $attributes)) {
            return $query->where("$table.id", '=', $attributes['id'])->get()->collect();
        }

        $key = $this->getViewAllCacheId();

        if ($level != 0) {
            $key = $this->getViewLevelCacheId();
            $query->where("$table.level", $level);
        }

        return Cache::rememberForever($key, function () use ($query, $table) {
            return $query->with($this->getRelation($table))
                ->orderBy("$table.ordering")
                ->get(["$table.*"])
                ->collect();
        });
    }

    protected function getRelation(string $table): array
    {
        return [
            'subCategories' => function (HasMany $q) use ($table) {
                $q->where("$table.is_active", MetaFoxConstant::IS_ACTIVE)
                    ->with($this->getRelation($table))
                    ->orderBy("$table.ordering");
            },
        ];
    }

    protected function getCategoryInactive()
    {
        /** @var \Illuminate\Database\Eloquent\Collection $categories */
        $categories = $this->getModel()->newQuery()
            ->select(['id', 'level', 'parent_id', 'is_active'])
            ->orderByDesc('level')->get();
        $depth      = $categories->first()->level;

        return Cache::rememberForever($this->getViewInactiveCacheId(), function () use ($categories, $depth) {
            $idsInactive = [];

            for ($level = 1; $level <= $depth; $level++) {
                $idsInactive = array_merge($categories->filter(function ($item) use ($level, $idsInactive) {
                    return ($item->is_active == MetaFoxConstant::IS_INACTIVE && $item->level == $level)
                        || in_array($item->parent_id, $idsInactive);
                })->pluck('id')->toArray(), $idsInactive);
            }

            return array_unique($idsInactive);
        });
    }

    public function viewForAdmin(User $context, array $attributes)
    {
        $parentId = Arr::get($attributes, 'parent_id');
        $search   = Arr::get($attributes, 'q');

        $query = $this->getModel()->newQuery()
            ->orderBy('ordering')
            ->with(['subCategories', 'parentCategory']);

        if ($search) {
            return $query->where('name', $this->likeOperator(), '%' . $search . '%')->get();
        }

        if (null === $parentId) {
            $query->whereNull('parent_id');
        }

        if (is_numeric($parentId)) {
            $query->where('parent_id', '=', $parentId);
        }

        return $query->get();
    }

    public function clearCache()
    {
        Cache::forget($this->getViewAllCacheId());
        Cache::forget($this->getViewLevelCacheId());
        Cache::forget($this->getViewInactiveCacheId());
        localCacheStore()->forget($this->getStructureCacheKey());
    }

    /**
     * @inheritDoc
     */
    public function incrementTotalItemCategories(?Model $category, int $totalItem): void
    {
        if (!$category instanceof Model) {
            return;
        }

        do {
            $total = $totalItem + $category->total_item;
            $category->update(['total_item' => $total]);
            $category = $category?->parentCategory;
        } while ($category);
    }

    /**
     * @inheritDoc
     */
    public function decrementTotalItemCategories(?Model $category, int $totalItem): void
    {
        if (!$category instanceof Model) {
            return;
        }

        do {
            $total = $category->total_item - $totalItem;
            $category->update(['total_item' => $total]);
            $category = $category?->parentCategory;
        } while ($category);
    }

    public function orderCategories(array $orderIds): bool
    {
        $categories = $this->getModel()->newQuery()
            ->whereIn('id', $orderIds)
            ->get()
            ->keyBy('id');

        if (!$categories->count()) {
            return true;
        }

        $ordering = 1;

        foreach ($orderIds as $orderId) {
            $category = $categories->get($orderId);

            if (!is_object($category)) {
                continue;
            }

            $category->update(['ordering' => $ordering++]);
        }

        return true;
    }

    public function toggleActive(int $id): Model
    {
        $item = $this->find($id);

        if (!$item instanceof Model) {
            abort(403, __p('core::validation.category_id.exists'));
        }

        if ($item->is_default) {
            abort(403, __p('core::validation.category_id.default'));
        }

        $item->update(['is_active' => $item->is_active ? 0 : 1]);

        $this->clearCache();

        return $item;
    }

    protected function getViewAllCacheId(): string
    {
        return $this->getModel()->getMorphClass() . '_get_all';
    }

    protected function getViewLevelCacheId(): string
    {
        return $this->getModel()->getMorphClass() . '_level';
    }

    protected function getViewInactiveCacheId(): string
    {
        return $this->getModel()->getMorphClass() . '_inactive';
    }

    protected function getDefaultCategoryParentIdsCacheKey(): string
    {
        return $this->getModel()->getMorphClass() . '_default_parent_ids';
    }

    protected function getStructureCacheKey(): string
    {
        return get_called_class() . '::getStructure';
    }

    protected function getCategoryRelation(?array $categoryIds): Builder
    {
        $table = $this->getModel()->getTable();

        $subQuery = $this->getModel()->newModelQuery()
            ->select('child.id as child_id', "$table.id as sub_id", 'child.parent_id as child_parent_id', 'child.level as child_level')
            ->join("$table as child", 'child.id', '=', "$table.parent_id");

        if (is_array($categoryIds) && count($categoryIds) > 0) {
            $subQuery->whereIn("$table.id", $categoryIds);
        }

        return $this->getModel()->newQuery()
            ->select(
                "$table.id as parent_id",
                's.sub_id as child_id',
                DB::raw("(case
                                   when $table.parent_id is null then s.child_level + $table.level
                                   when s.sub_id = $table.id then 1
                                   else s.child_level end) as depth")
            )
            ->joinSub(
                $subQuery,
                's',
                function (JoinClause $joinClause) use ($table) {
                    $joinClause->on('s.child_id', '=', "$table.id");
                    $joinClause->orWhere('s.child_parent_id', '=', DB::raw("$table.id"));
                    $joinClause->orWhere('s.sub_id', '=', DB::raw("$table.id"));
                }
            );
    }

    /**
     * @param  Model $category
     * @param  int   $newCategoryId
     * @return void
     */
    protected function changeParentCategory(Model $category, int $newCategoryId): void
    {
        $totalItem = $category->total_item;
        $parent    = $category?->parentCategory;

        $this->decrementTotalItemCategories($parent, $totalItem);
        $newCategory = $this->find($newCategoryId);

        if ($newCategory->level >= $category->level && $category->subCategories()->exists()) {
            abort(403, json_encode([
                'title'   => __p('core::phrase.content_is_not_available'),
                'message' => __p('core::phrase.it_is_not_allowed_to_move_this_category_and_subcategories_to_the_selected_category', [
                    'value' => MetaFoxConstant::MAX_CATEGORY_LEVEL,
                ]),
            ]));
        }

        $category->update([
            'parent_id' => $newCategory->entityId(),
            'level'     => $newCategory->level + 1,
        ]);

        $category->refresh();

        //update new level for category children
        $this->getModel()->newQuery()
            ->where('parent_id', $category->entityId())
            ->update(['level' => $category->level + 1]);

        $this->deleteCategoryRelations(category: $category);

        $this->createCategoryRelationFor(category: $category);
        $this->clearCache();

        $this->incrementTotalItemCategories($newCategory, $totalItem);
    }

    /**
     * @param  Model $category
     * @return void
     */
    public function createCategoryRelationFor(Model $category): void
    {
        $model = $this->getRelationModel();

        $categoryIds = array_merge([$category->entityId()], $category->subCategories()->pluck('id')->toArray());

        if (!$category->parentCategory()->exists()) {
            $model->fill([
                'parent_id' => $category->entityId(),
                'child_id'  => $category->entityId(),
                'depth'     => $category->level,
            ])->save();

            return;
        }

        $query = $this->getCategoryRelation($categoryIds);

        $model->newQuery()->insertUsing(['parent_id', 'child_id', 'depth'], $query);
    }

    /**
     * @return void
     */
    public function createCategoryRelation(): void
    {
        $categoryIds = null;
        $model       = $this->getRelationModel();

        $query = $this->getCategoryRelation($categoryIds);

        $model->newQuery()->insertUsing(['parent_id', 'child_id', 'depth'], $query);
    }

    /**
     * @param  Model $category
     * @return void
     */
    public function deleteCategoryRelations(Model $category): void
    {
        $categoryId = $category->entityId();

        $model = $this->getRelationModel();

        $model->newQuery()->whereIn('child_id', $this->getChildrenIds($categoryId))->delete();
        $model->newQuery()->whereIn('parent_id', $this->getChildrenIds($categoryId))->delete();
    }

    /**
     * @param  int   $categoryId
     * @return array
     */
    public function getChildrenIds(int $categoryId): array
    {
        $model = $this->getRelationModel();

        return $model->newQuery()->where('parent_id', $categoryId)
            ->pluck('child_id')->toArray();
    }

    /**
     * @param  int   $categoryId
     * @return array
     */
    public function getParentIds(int $categoryId): array
    {
        $model = $this->getRelationModel();

        return $model->newQuery()->where('child_id', $categoryId)
            ->whereNot('parent_id', $categoryId)
            ->pluck('parent_id')->toArray();
    }

    /**
     * @inheritDoc
     */
    public function createTopLevelCategoryRelation(): void
    {
        $query = $this->getModel()->newQuery()
            ->select('id as parent_id', 'id as child_id', 'level')
            ->whereNull('parent_id');

        $this->getRelationModel()->newQuery()->insertUsing(['parent_id', 'child_id', 'depth'], $query);
    }

    public function migrateCategoryRelationAfterImport(string $tableName): void
    {
        if (!Schema::hasTable($tableName)) {
            return;
        }

        $this->getRelationModel()::truncate();
        $this->createTopLevelCategoryRelation();
        $this->createCategoryRelation();
    }
}
