<?php

namespace MetaFox\User\Support;

use Exception;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Http\Request as SystemRequest;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection as CollectionSupport;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Request;
use Illuminate\Support\Str;
use MetaFox\Authorization\Models\Role;
use MetaFox\Core\Support\Facades\Country;
use MetaFox\Localize\Repositories\TimezoneRepositoryInterface;
use MetaFox\Platform\Contracts\HasUserProfile;
use MetaFox\Platform\Contracts\User as ContractUser;
use MetaFox\Platform\Facades\LoadReduce;
use MetaFox\Platform\Facades\Settings;
use MetaFox\Platform\MetaFox;
use MetaFox\Platform\MetaFoxConstant;
use MetaFox\Platform\UserRole;
use MetaFox\User\Contracts\UserContract;
use MetaFox\User\Models\User as UserModel;
use MetaFox\User\Models\UserActivity;
use MetaFox\User\Models\UserGender;
use MetaFox\User\Models\UserProfile;
use MetaFox\User\Repositories\Contracts\UserRepositoryInterface;
use MetaFox\User\Repositories\UserRelationRepositoryInterface;
use MetaFox\User\Support\Facades\UserPrivacy;
use MetaFox\User\Support\Facades\UserValue;
use MetaFox\User\Support\User as Support;
use MetaFox\User\Traits\UserLocationTrait;

/**
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 * @SuppressWarnings(PHPMD.ExcessiveClassComplexity)
 */
class User implements UserContract
{
    // todo DO NOT MIX TRAIT HERE.
    use UserLocationTrait;

    public const MENTION_REGEX                 = '^\[user=(.*?)\]^';
    public const DATE_OF_BIRTH_DONT_SHOW       = 1;
    public const DATE_OF_BIRTH_SHOW_DAY_MONTH  = 2;
    public const DATE_OF_BIRTH_SHOW_AGE        = 3;
    public const DATE_OF_BIRTH_SHOW_ALL        = 4;
    public const AUTO_APPROVED_TAGGER_POST     = 0;
    public const NOT_AUTO_APPROVED_TAGGER_POST = 1;
    public const DISPLAY_FULL_NAME             = 'full_name';
    public const DISPLAY_USER_NAME             = 'user_name';
    public const DISPLAY_BOTH                  = 'both';
    public const GENERATED_USER_NAME           = 'user-%s-%s';

    /**
     * Profile/Account setting name.
     */
    public const AUTO_APPROVED_TAGGED_SETTING = 'user_auto_add_tagger_post';
    public const AUTO_PLAY_VIDEO_SETTING      = 'user_auto_play_videos';
    public const THEME_TYPE_SETTING           = 'profile_theme_type';
    public const THEME_ID_SETTING             = 'profile_theme_id';

    public const THEME_PREFERENCE_NAMES = [
        self::THEME_TYPE_SETTING,
        self::THEME_ID_SETTING,
    ];

    private UserRepositoryInterface $repository;

    public function __construct(
        UserRepositoryInterface $repository,
        protected UserRelationRepositoryInterface $relationRepository
    ) {
        $this->repository = $repository;
    }

    public function isBan(int $userId): bool
    {
        return $this->repository->isBanned($userId);
    }

    public function getFriendship(ContractUser $user, ContractUser $targetUser): ?int
    {
        // todo how to reduce request from users.
        return LoadReduce::get(
            sprintf('friend::friendship(user:%s,owner:%s)', $user->id, $targetUser->id),
            function () use ($user, $targetUser) {
                $friendship = app('events')->dispatch('friend.get_friend_ship', [$user, $targetUser], true);
                if (is_int($friendship)) {
                    return $friendship;
                }

                return $user->entityId() == $targetUser->entityId() ? 5 : 6;
            }
        );
    }

    public function getGuestUser(): Authenticatable
    {
        $user            = new UserModel();
        $user->id        = MetaFoxConstant::GUEST_USER_ID;
        $user->user_name = 'guest';
        $user->full_name = 'Guest';

        try {
            // setup wizard issues auth_roles still not exists.
            $guestUser    = Role::findById(UserRole::GUEST_USER);
            $guestProfile = new UserProfile(['id' => $user->id]);
            $user->setRelation('roles', new Collection([$guestUser]));
            $user->setRelation('profile', $guestProfile);
        } catch (Exception) {
        }

        return $user;
    }

    public function getGender(UserProfile $profile): ?string
    {
        $cacheId = sprintf('gender_phrase_of_%s', $profile->gender_id);

        $phrase = Cache::rememberForever($cacheId, function () use ($profile) {
            if ($profile->gender instanceof UserGender) {
                return $profile->gender->phrase;
            }

            return '_';
        });

        return '_' != $phrase ? __p($phrase) : null;
    }

    public function getBirthday(ContractUser $user): ?string
    {
        $formatValue = UserValue::getUserValueSettingByName($user, 'user_profile_date_of_birth_format');
        $birthday    = $user?->profile?->birthday;

        if (!is_string($birthday)) {
            return null;
        }

        $time = Carbon::createFromFormat('Y-m-d', $birthday);

        if ($time === false) {
            return $birthday;
        }

        return match ($formatValue) {
            self::DATE_OF_BIRTH_SHOW_DAY_MONTH => $time->translatedFormat('m-d'),
            self::DATE_OF_BIRTH_SHOW_ALL       => $time->translatedFormat('Y-m-d'),
            default                            => null,
        };
    }

    public function getUserAge(?string $birthday): ?int
    {
        if (!is_string($birthday)) {
            return null;
        }

        $time = Carbon::createFromFormat('Y-m-d', $birthday);

        if ($time === false) {
            return null;
        }

        return $time->age;
    }

    public function getFullBirthdayFormat(): array
    {
        return [
            'F j, Y', 'Y-m-d', 'm/d/Y', 'd/m/Y',
        ];
    }

    public function getMonthDayBirthdayFormat(): array
    {
        return [
            'F j', 'm-d', 'm/d', 'd/m',
        ];
    }

    public function splitName(string $name): array
    {
        // @todo need to test after.
        $firstName = $middleName = $lastName = '';

        $name = trim($name);

        preg_match('/^([\w-]+)(?:\s+)?(.*?)(?:\s+)?([\w-]+)?$/', $name, $matches);

        if (count($matches)) {
            /**
             * Remove full matching of string
             */
            array_shift($matches);

            $firstName = array_shift($matches);

            $lastName  = array_pop($matches);

            if (count($matches)) {
                $middleName = array_shift($matches);
            }
        }

        return [$lastName, $firstName, $middleName];
    }

    public function getLastName(string $name): string
    {
        [$lastName] = $this->splitName($name);

        return $lastName;
    }

    public function getFirstName(string $name): string
    {
        [, $firstName, $middleName] = $this->splitName($name);

        if ($middleName && $middleName != '') {
            return $firstName . ' ' . $middleName;
        }

        return $firstName;
    }

    public function getShortName(string $name): string
    {
        $lastName  = self::getLastName($name);
        $firstName = self::getFirstName($name);

        $lastNameString  = ((isset($lastName[0])) ? $lastName[0] : '');
        $firstNameString = ((isset($firstName[0])) ? $firstName[0] : '');

        if (!$lastNameString) {
            return Str::upper($firstNameString . ((isset($firstName[1])) ? $firstName[1] : ''));
        }

        $shortName = $firstNameString . $lastNameString;

        return Str::upper($shortName);
    }

    /**
     * @inherhitDoc
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     */
    public function getSummary(ContractUser $context, ContractUser $user): ?string
    {
        if (!UserPrivacy::hasAccess($context, $user, 'profile.view_profile')) {
            return null;
        }

        if (!UserPrivacy::hasAccess($context, $user, 'profile.profile_info')) {
            return null;
        }

        $summary = [];
        $profile = $user->profile;

        if (!$profile instanceof UserProfile) {
            return null;
        }
        $address = $this->getAddress($context, $user);
        if ($address) {
            $summary[] = $address;
        }

        $summary[] = Str::limit($profile->about_me, 155);

        return implode('. ', $summary);
    }

    public function getAge(ContractUser $user): ?int
    {
        $age = null;

        $birthdaySetting = UserValue::getUserValueSettingByName($user, 'user_profile_date_of_birth_format');

        if (Support::DATE_OF_BIRTH_SHOW_ALL != $birthdaySetting) {
            return null;
        }
        $birthday = $user?->profile?->birthday;
        try {
            if (!empty($birthday)) {
                $age = Carbon::parse($birthday)->age;
            }
        } catch (Exception $e) {
            // Just silent.
        }

        return $age;
    }

    public function getNewAgePhrase(ContractUser $user): ?string
    {
        if (!$user instanceof UserModel) {
            return null;
        }

        $birthdaySetting = UserValue::getUserValueSettingByName($user, 'user_profile_date_of_birth_format');

        if (!in_array($birthdaySetting, [Support::DATE_OF_BIRTH_SHOW_ALL, Support::DATE_OF_BIRTH_SHOW_AGE])) {
            return null;
        }

        $birthday = $user->profile->birthday;
        $age      = $this->getAge($user);
        $newAge   = $age + 1;

        $date = Carbon::parse($birthday)->format('m-d');
        $now  = Carbon::make(MetaFox::clientDate());

        if ($date == $now->format('m-d')) {
            $newAge = $age;
        }

        return __p('user::phrase.years_old', ['year' => $newAge]);
    }

    public function getTimeZoneForForm(): array
    {
        $timezones     = [];
        $timezonesData = resolve(TimezoneRepositoryInterface::class)->getTimeZones();

        if ($timezonesData) {
            $timezones = collect($timezonesData[0])->map(function ($value) {
                return [
                    'id'   => $value['id'],
                    'name' => "{$value['name']} ({$value['diff_from_gtm']})",
                ];
            })->toArray();
        }

        return $timezones;
    }

    public function getTimeZoneNameById(int $id): ?string
    {
        if (0 == $id) {
            return null;
        }

        $timezonesData = resolve(TimezoneRepositoryInterface::class)->getTimeZones();

        if ($timezonesData) {
            if (isset($timezonesData[0][$id])) {
                return $timezonesData[0][$id]['name'];
            }
        }

        return null;
    }

    public function getUsersByRoleId(int $roleId): ?CollectionSupport
    {
        return $this->repository->getUsersByRoleId($roleId);
    }

    public function getMentions(string $content): array
    {
        $userIds = [];
        try {
            preg_match_all(self::MENTION_REGEX, $content, $matches);
            $userIds = array_unique($matches[1]);
        } catch (Exception $e) {
            // Silent.
        }

        return $userIds;
    }

    public function getPossessiveGender(?UserGender $gender): string
    {
        $defaultGender = __p('core::phrase.their');

        if (null === $gender) {
            return $defaultGender;
        }

        switch ($gender->entityId()) {
            case MetaFoxConstant::GENDER_MALE:
                $gender = __p('core::phrase.his');
                break;
            case MetaFoxConstant::GENDER_FEMALE:
                $gender = __p('core::phrase.her');
                break;
            default:
                $gender = $defaultGender;
        }

        return $gender;
    }

    public function updateLastLogin(?ContractUser $context): bool
    {
        // check login as guest.
        if (!$context) {
            return false;
        }

        if ($context instanceof HasUserProfile && !$context->isGuest()) {
            return UserActivity::query()
                ->where('id', $context->entityId())
                ->update([
                    'last_login'      => now(),
                    'last_ip_address' => Request::ip(),
                ]);
        }

        return false;
    }

    public function updateLastActivity(ContractUser $context): bool
    {
        if ($context instanceof HasUserProfile && $context->id) {
            return UserActivity::query()
                ->where('id', $context->id)
                ->where('last_activity', '<=', Carbon::now()->subMinutes(5))
                ->update([
                    'last_activity' => now(),
                ]);
        }

        return false;
    }

    public function updateInvisibleMode(ContractUser $context, int $isInvisible): UserModel
    {
        return $this->repository->updateUser($context, $context->entityId(), ['is_invisible' => $isInvisible]);
    }

    /**
     * @param ContractUser $user
     *
     * @return array<int, mixed>
     */
    public function getNotificationSettingsByChannel(ContractUser $user, string $channel): array
    {
        $settings = app('events')
            ->dispatch('notification.get_notification_settings_by_channel', [$user, $channel], true);

        return !empty($settings) ? $settings : [];
    }

    /**
     * @param ContractUser      $context
     * @param array<string,int> $attributes
     *
     * @return bool
     */
    public function updateNotificationSettingsByChannel(ContractUser $context, array $attributes): bool
    {
        return app_active('metafox/notification')
            && app('events')->dispatch(
                'notification.update_email_notification_settings',
                [$context, $attributes],
                true
            );
    }

    public function hasPendingSubscription(SystemRequest $request, ContractUser $user, bool $isMobile = false): ?array
    {
        return app('events')->dispatch('subscription.check_pending_user', [$request, $user, $isMobile], true);
    }

    /**
     * @inheritDoc
     */
    public function getAddress(ContractUser $context, ContractUser $user): ?string
    {
        if (!$this->canViewLocation($context, $user)) {
            return null;
        }

        if (!$user->profile instanceof UserProfile) {
            return null;
        }

        $profile = $user->profile;
        $country = $state = '';
        $city    = $profile->city_location ?? '';

        if ($profile->country_iso) {
            $country = Country::getCountryName($profile->country_iso);
        }

        if ($country && $profile->country_state_id) {
            $state = Country::getCountryStateName($profile->country_iso, $profile->country_state_id);
        }
        $locations = array_filter([$city, $state, $country]);

        return $locations ? __p('user::phrase.lives_in', ['locations' => implode(', ', $locations)]) : null;
    }

    /**
     * @inheritDoc
     */
    public function isFollowing(ContractUser $context, ContractUser $user): bool
    {
        if (!app('events')->dispatch('follow.is_follow', [$context, $user], true)) {
            return false;
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public function totalFollowers(ContractUser $user): int
    {
        return app('events')->dispatch('follow.get_total_follow', [$user, MetaFoxConstant::VIEW_FOLLOWER], true) ?? 0;
    }

    /**
     * @inheritDoc
     */
    public function getBirthdayPhrase(ContractUser $context, ContractUser $user): string
    {
        if (!$user instanceof UserModel) {
            return MetaFoxConstant::EMPTY_STRING;
        }

        $birthdaySetting = UserValue::getUserValueSettingByName($user, 'user_profile_date_of_birth_format');
        $birthday        = $user->profile->birthday;

        if ($birthday === null) {
            return MetaFoxConstant::EMPTY_STRING;
        }

        $date   = Carbon::parse($birthday)->format('m-d');
        $format = match ($birthdaySetting) {
            self::DATE_OF_BIRTH_SHOW_DAY_MONTH => Settings::get('user.user_dob_month_day', 'F j'),
            self::DATE_OF_BIRTH_SHOW_ALL       => Settings::get('user.user_dob_month_day_year', 'F j, Y'),
            self::DATE_OF_BIRTH_SHOW_AGE, self::DATE_OF_BIRTH_DONT_SHOW => MetaFoxConstant::EMPTY_STRING,
            default => 'F jS'
        };

        $now = Carbon::make(MetaFox::clientDate());

        $userEntity = $user->userEntity()->first();

        if ($date == $now->translatedFormat('m-d')) {
            if ($context->entityId() == $user->entityId()) {
                return __p('user::phrase.today');
            }

            return __p('user::phrase.value_is_gender_birthday', [
                'value'  => __p('user::phrase.today'),
                'gender' => $userEntity->possessive_gender,
            ]);
        }

        if ($date == $now->tomorrow()->translatedFormat('m-d') && $context->entityId() != $user->entityId()) {
            return __p('user::phrase.value_is_gender_birthday', [
                'value'  => __p('user::phrase.tomorrow'),
                'gender' => $userEntity->possessive_gender,
            ]);
        }

        return __p('user::phrase.birthday_format', [
            'value' => Carbon::parse($user->profile->birthday)->locale($context->preferredLocale())->format($format),
        ]);
    }

    /**
     * @param  ContractUser         $user
     * @return array<string, mixed>
     */
    public function getVideoSettings(ContractUser $user): array
    {
        $settings = [
            self::AUTO_PLAY_VIDEO_SETTING,
        ];

        $data = [];

        foreach ($settings as $setting) {
            $data[$setting] = UserValue::getUserValueSettingByName($user, $setting);
        }

        return $data;
    }
}
