<?php

namespace MetaFox\User\Http\Resources\v1\UserInactive\Admin;

use MetaFox\Authorization\Repositories\Contracts\RoleRepositoryInterface;
use MetaFox\Form\AbstractForm;
use MetaFox\Form\Builder as Builder;
use MetaFox\Form\Constants as MetaFoxForm;
use MetaFox\Platform\MetaFoxConstant;
use MetaFox\User\Models\UserGender as Model;
use MetaFox\User\Support\Browse\Scopes\User\SortScope;
use MetaFox\User\Support\Browse\Scopes\User\StatusScope;
use MetaFox\Yup\Yup;

/**
 * --------------------------------------------------------------------------
 * Form Configuration
 * --------------------------------------------------------------------------
 * stub: /packages/resources/edit_form.stub.
 */

/**
 * Class SearchUserInactiveForm.
 * @property ?Model $resource
 * @ignore
 * @codeCoverageIgnore
 * @driverName user.user_gender.search
 */
class SearchUserInactiveForm extends AbstractForm
{
    protected function prepare(): void
    {
        $this->title(__p('user::phrase.inactive_user'))
            ->action('/admincp/user/inactive')
            ->acceptPageParams([
                'q', 'email', 'group', 'status', 'gender',
                'postal_code', 'country_state_id', 'country', 'day',
                'age_from', 'age_to', 'sort', 'ip_address', 'phone_number',
            ])
            ->submitAction(MetaFoxForm::FORM_SUBMIT_ACTION_SEARCH)
            ->setValue([
                'group'  => null,
                'status' => StatusScope::STATUS_DEFAULT,
                'sort'   => SortScope::SORT_DEFAULT,
                'gender' => 0,
                'day'    => 7,
            ]);
    }

    protected function initialize(): void
    {
        $basic = $this->addBasic()
            ->asHorizontal()
            ->sxContainer(['alignItems' => 'unset']);

        $basic->addFields(
            Builder::text('day')
                ->forAdminSearchForm()
                ->required()
                ->label(__p('user::phrase.day'))
                ->setAttribute('type', 'number')
                ->description(__p('user::phrase.number_day'))
                ->maxWidth(380)
                ->yup(Yup::number()->required()->unint()),
            Builder::text('q')
                ->forAdminSearchForm()
                ->label(__p('core::phrase.search')),
            Builder::text('email')
                ->forAdminSearchForm()
                ->label(__p('user::phrase.email')),
            Builder::text('phone_number')
                ->forAdminSearchForm()
                ->label(__p('core::phrase.phone_number')),
            Builder::choice('group')
                ->forAdminSearchForm()
                ->label(__p('authorization::phrase.roles'))
                ->options($this->getGroupOptions()),
            Builder::gender()
                ->label(__p('user::phrase.user_gender'))
                ->forAdminSearchForm(),
            Builder::text('postal_code')
                ->forAdminSearchForm()
                ->label(__p('user::phrase.zip_postal_code'))
                ->placeholder('- - - - - -'),
            Builder::choice('status')
                ->fullWidth(false)
                ->sizeSmall()
                ->marginDense()
                ->width(220)
                ->showWhen(['and', ['truthy', 'view_more']])
                ->label(__p('user::phrase.show_members'))
                ->options($this->getStatusOptions()),
            Builder::choice('age_from')
                ->forAdminSearchForm()
                ->showWhen(['and', ['truthy', 'view_more']])
                ->label(__p('user::phrase.age_group_from'))
                ->options($this->getAgeOptions()),
            Builder::choice('age_to')
                ->forAdminSearchForm()
                ->showWhen(['and', ['truthy', 'view_more']])
                ->label(__p('user::phrase.age_group_to'))
                ->options($this->getAgeOptions()),
            Builder::choice('sort')
                ->forAdminSearchForm()
                ->showWhen(['and', ['truthy', 'view_more']])
                ->label(__p('user::phrase.sort_results_by'))
                ->options(SortScope::getSortOptions()),
            Builder::text('city')
                ->forAdminSearchForm()
                ->label(__p('localize::country.city'))
                ->showWhen(['and', ['truthy', 'view_more']])
                ->placeholder(__p('localize::country.filter_by_city')),
            Builder::text('ip_address')
                ->forAdminSearchForm()
                ->showWhen(['and', ['truthy', 'view_more']])
                ->label(__p('user::phrase.ip_address')),
            Builder::countryState('country')
                ->sizeSmall()
                ->maxWidth(220)
                ->forAdminSearchForm()
                ->valueType('array')
                ->showWhen(['and', ['truthy', 'view_more']])
                ->setAttribute('countryFieldName', 'country')
                ->setAttribute('stateFieldName', 'country_state_id')
                ->inline(),
        );

        $basic->addFields(
            Builder::submit()
                ->forAdminSearchForm()
                ->label(__p('user::phrase.get_inactive_member')),
            Builder::submitAction('process_mailing_all')
                ->forAdminSearchForm()
                ->label(__p('user::phrase.process_mailing_job_to_all_inactive_members'))
                ->variant('contained')
                ->sizeMedium()
                ->customAction([
                    'module_name'   => 'user',
                    'resource_name' => 'user_inactive',
                    'action_name'   => 'processMailingAll',
                ]),
            Builder::clearSearchForm()
                ->label(__p('core::phrase.reset'))
                ->marginDense()
                ->align('right')
                ->excludeFields(['view_more'])
                ->sxFieldWrapper([
                    'p'              => 1,
                    'justifyContent' => 'center',
                    'alignItems'     => 'center,',
                ]),
            Builder::viewMore('view_more')
                ->marginDense()
                ->sxFieldWrapper([
                    'p'              => 1,
                    'justifyContent' => 'center',
                    'alignItems'     => 'center,',
                ]),
        );
    }

    private function getGroupOptions(): array
    {
        return array_merge(
            [
                [
                    'label' => __p('core::phrase.any'),
                    'value' => null,
                ],
            ],
            resolve(RoleRepositoryInterface::class)->getRoleOptions()
        );
    }

    public function getAgeOptions(): array
    {
        return array_map(function (int $value) {
            return [
                'label' => $value,
                'value' => $value,
            ];
        }, range(4, 121));
    }

    protected function getStatusOptions(): array
    {
        return array_filter(StatusScope::getStatusOptions(), function ($status) {
            return $status['value'] != MetaFoxConstant::STATUS_ONLINE;
        });
    }
}
