<?php

namespace MetaFox\User\Http\Resources\v1\User;

use Illuminate\Auth\AuthenticationException;
use Illuminate\Support\Carbon;
use MetaFox\Core\Support\Facades\Country;
use MetaFox\Form\AbstractField;
use MetaFox\Form\GenderTrait;
use MetaFox\Form\Mobile\Builder;
use MetaFox\Form\Mobile\MobileForm as AbstractForm;
use MetaFox\Form\Section;
use MetaFox\Platform\Facades\Settings;
use MetaFox\Platform\MetaFoxConstant;
use MetaFox\Platform\MetaFoxFileType;
use MetaFox\Platform\Rules\MetaFoxPasswordFormatRule;
use MetaFox\Profile\Repositories\ProfileRepositoryInterface;
use MetaFox\RegexRule\Support\Facades\Regex;
use MetaFox\User\Support\User as UserSupport;
use MetaFox\Yup\StringShape;
use MetaFox\Yup\Yup;

/**
 * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
 *
 * @driverType form-mobile
 * @driverName user.register
 * @preload    1
 */
class UserRegisterMobileForm extends AbstractForm
{
    use GenderTrait;

    /**
     * @return void
     */
    public function boot(): void
    {
    }

    protected function prepare(): void
    {
        $this->title(__p('user::phrase.create_account'))
            ->action('/register')
            ->asPost()
            ->submitAction('user/register');
    }

    /**
     * @throws AuthenticationException
     */
    public function initialize(): void
    {
        $isBasicFieldRequired = Settings::get('user.require_basic_field', false);
        $basic = $this->addBasic();

        if (Settings::get('user.force_user_to_upload_on_sign_up', false)) {
            $basic->addField(
                Builder::avatarUpload('user_profile')
                    ->required()
                    ->accept(file_type()->getMimeTypeFromType(MetaFoxFileType::PHOTO_TYPE, false))
                    ->label(__p('user::phrase.profile_image'))
                    ->placeholder(__p('user::phrase.profile_image'))
                    ->description(__p('user::phrase.profile_image_desc'))
                    ->yup(
                        Yup::object()->addProperty(
                            'base64',
                            Yup::string()->required(__p('validation.field_is_a_required_field', [
                                'field' => __p('user::phrase.profile_image'),
                            ]))
                        )
                    )
            );
        }

        $this->handleFullNameField($basic);
        $this->handleUserNameField($basic);
        match (Settings::get('user.enable_phone_number_registration')) {
            true    => $this->handleEmailAndPhoneNumberFields($basic),
            default => $this->handleEmailFields($basic),
        };

        $basic->addField(
            Builder::password('password')
                ->autoComplete('off')
                ->marginNormal()
                ->label(__p('user::phrase.password'))
                ->placeholder(__p('user::phrase.password'))
                ->returnKeyType('next')
                ->required()
                ->minLength(Settings::get('user.minimum_length_for_password', 8))
                ->maxLength(Settings::get('user.maximum_length_for_password', 30))
                ->yup($this->getPasswordValidate()),
        );

        if (Settings::get('user.signup_repeat_password')) {
            $basic->addFields(
                Builder::password('password_confirmation')
                    ->autoComplete('off')
                    ->marginNormal()
                    ->label(__p('user::phrase.confirm_password'))
                    ->placeholder(__p('user::phrase.confirm_password'))
                    ->returnKeyType('next')
                    ->noFeedback(true)
                    ->required()
                    ->minLength(Settings::get('user.minimum_length_for_password', 8))
                    ->maxLength(Settings::get('user.maximum_length_for_password', 30))
                    ->yup(
                        Yup::string()
                            ->required()
                            ->setError('required', __p('validation.this_field_is_a_required_field'))
                            ->setError('typeError', __p('validation.password_is_a_required_field')),
                    ),
            );
        }
        $this->addBirthdayField($basic, $isBasicFieldRequired);
        $this->addGenderField($basic, $isBasicFieldRequired);
        $this->addLocationField($basic, $isBasicFieldRequired);
        $this->addCityField($basic);

        app('events')->dispatch('user.registration.extra_fields.build', [$basic]);

        $this->handleCustomField($basic);

        if (Settings::get('user.new_user_terms_confirmation')) {
            $basic->addField(
                Builder::checkbox('agree')
                    ->label(__p('user::phrase.agree_field_label_embed', [
                        'term_of_use_link' => 'term-of-use',
                        'policy_link'      => 'policy',
                    ]))
                    ->setAttribute('isReverse', true)
                    ->variant('standard-inlined-end')
                    ->returnKeyType('next')
                    ->required()
                    ->yup(
                        Yup::number()
                            ->required()
                            ->setError('required', __p('validation.agree_field_is_a_required_field'))
                            ->setError('typeError', __p('validation.agree_field_is_a_required_field'))
                    )
            );
        }

        $footer = $this->addFooter([
            'variant' => 'horizontal',
            'style'   => [
                'justifyContent' => 'space-between',
            ],
        ]);

        $footer->addFields(
            Builder::submit()
                ->sizeLarge()
                ->marginDense()
                ->label(__p('user::phrase.create_account')),
            Builder::linkButton()
                ->link('/login')
                ->marginDense()
                ->fullWidth()
                ->variant('standard')
                ->sizeLarge()
                ->label(__p('user::phrase.already_had_an_account'))
        );
    }

    /**
     * @throws AuthenticationException
     */
    public function initializeFlatten(): void
    {
        $isBasicFieldRequired = Settings::get('user.require_basic_field', false);
        $basic = $this->addBasic();

        if (Settings::get('user.force_user_to_upload_on_sign_up', false)) {
            $basic->addField(
                Builder::avatarUpload('user_profile')
                    ->variant('standard-outlined')
                    ->required()
                    ->accept(file_type()->getMimeTypeFromType(MetaFoxFileType::PHOTO_TYPE, false))
                    ->label(__p('user::phrase.profile_image'))
                    ->placeholder(__p('user::phrase.profile_image'))
                    ->description(__p('user::phrase.profile_image_desc'))
                    ->yup(
                        Yup::object()->addProperty(
                            'base64',
                            Yup::string()->required(__p('validation.field_is_a_required_field', [
                                'field' => __p('user::phrase.profile_image'),
                            ]))
                        )
                    )
            );
        }

        $this->handleFullNameFieldFlatten($basic);
        $this->handleUserNameFieldFlatten($basic);
        match (Settings::get('user.enable_phone_number_registration')) {
            true    => $this->handleEmailAndPhoneNumberFieldsFlatten($basic),
            default => $this->handleEmailFieldsFlatten($basic),
        };

        $basic->addField(
            Builder::password('password')
                ->autoComplete('off')
                ->marginNone()
                ->variant('standard-outlined')
                ->label(__p('user::phrase.password'))
                ->placeholder(__p('user::phrase.password'))
                ->returnKeyType('next')
                ->required()
                ->minLength(Settings::get('user.minimum_length_for_password', 8))
                ->maxLength(Settings::get('user.maximum_length_for_password', 30))
                ->yup($this->getPasswordValidate()),
        );

        if (Settings::get('user.signup_repeat_password')) {
            $basic->addFields(
                Builder::password('password_confirmation')
                    ->autoComplete('off')
                    ->variant('standard-outlined')
                    ->label(__p('user::phrase.confirm_password'))
                    ->placeholder(__p('user::phrase.confirm_password'))
                    ->returnKeyType('next')
                    ->noFeedback(true)
                    ->required()
                    ->yup(
                        Yup::string()
                            ->required()
                            ->setError('required', __p('validation.this_field_is_a_required_field'))
                            ->setError('typeError', __p('validation.password_is_a_required_field')),
                    ),
            );
        }
        $this->addBirthdayField($basic, $isBasicFieldRequired);
        $this->addGenderField($basic, $isBasicFieldRequired);
        $this->addLocationField($basic, $isBasicFieldRequired);
        $this->addCityField($basic);

        app('events')->dispatch('user.registration.extra_fields.build', [$basic]);

        $this->handleCustomField($basic);

        if (Settings::get('user.new_user_terms_confirmation')) {
            $basic->addField(
                Builder::checkbox('agree')
                    ->label(__p('user::phrase.agree_field_label_embed', [
                        'term_of_use_link' => 'term-of-use',
                        'policy_link'      => 'policy',
                    ]))
                    ->setAttribute('isReverse', true)
                    ->variant('standard-inlined-end')
                    ->returnKeyType('next')
                    ->required()
                    ->yup(
                        Yup::number()
                            ->required()
                            ->setError('required', __p('validation.agree_field_is_a_required_field'))
                            ->setError('typeError', __p('validation.agree_field_is_a_required_field'))
                    )
            );
        }

        $footer = $this->addFooter([
            'variant' => 'horizontal',
            'style'   => [
                'justifyContent' => 'space-between',
            ],
        ]);

        $footer->addFields(
            Builder::submit()
                ->sizeLarge()
                ->marginDense()
                ->label(__p('user::phrase.create_account')),
            Builder::linkButton()
                ->link('/login')
                ->marginDense()
                ->fullWidth()
                ->variant('standard')
                ->sizeLarge()
                ->label(__p('user::phrase.already_had_an_account'))
        );
    }

    protected function getPasswordValidate(): StringShape
    {
        $passwordValidate = Yup::string()
            ->required()
            ->setError('required', __p('validation.this_field_is_a_required_field'))
            ->setError('typeError', __p('validation.password_is_a_required_field'))
            ->setError('minLength', __p('validation.field_must_be_at_least_min_length_characters', [
                'field'     => '${path}',
                'minLength' => '${min}',
            ]));

        $passwordRule = new MetaFoxPasswordFormatRule();

        $passwordValidate->matchesArray($passwordRule->getFormRules(), $passwordRule->message());

        return $passwordValidate;
    }

    protected function addBirthdayField(Section $basic, bool $isBasicFieldRequired): void
    {
        if (!Settings::get('user.enable_date_of_birth', false)) {
            return;
        }

        $minYear = Settings::get('user.date_of_birth_start', 1900);
        $maxYear = Settings::get('user.date_of_birth_end', Carbon::now()->year);
        $minDate = Carbon::create($minYear);
        $maxDate = Carbon::create($maxYear);
        $minDateString = $minDate ? $minDate->toDateString() : $minYear;
        $maxDateString = $maxDate ? $maxDate->endOfYear()->toDateString() : $maxYear;
        $birthdayMessage = __p('validation.invalid_date_of_birth_between', [
            'date_start' => $minDateString,
            'date_end'   => $maxDateString,
        ]);
        $validation = Yup::date()
            ->minYear((string)$minYear, $birthdayMessage)
            ->maxYear((string)$maxYear, $birthdayMessage)
            ->setError('typeError', __p('core::phrase.invalid_date'));

        $basic->addField(
            Builder::birthday('birthday')
                ->label(__p('user::phrase.birthday'))
                ->required($isBasicFieldRequired)
                ->setAttribute('minDate', $minDateString)
                ->setAttribute('maxDate', $maxDateString)
                ->yup(
                    $isBasicFieldRequired
                        ? $validation->required(__p('user::validation.birthday_is_a_required_field'))
                        : $validation
                )
        );
    }

    /**
     * @throws AuthenticationException
     */
    protected function addGenderField(Section $basic, bool $isBasicFieldRequired): void
    {
        if (!Settings::get('user.enable_gender', false)) {
            return;
        }

        $context = user();
        $validation = Yup::number();

        $basic->addFields(
            Builder::choice('gender')
                ->label(__p('user::phrase.i_am'))
                ->required($isBasicFieldRequired)
                ->options($this->getDefaultGenders($context))
                ->yup(
                    $isBasicFieldRequired
                        ? $validation->required(__p('user::validation.gender_is_a_required_field'))
                        : $validation
                ),
            Builder::choice('custom_gender')
                ->label(__p('user::phrase.custom_gender'))
                ->showWhen(['and', ['eq', 'gender', 0]])
                ->requiredWhen(['and', ['eq', 'gender', 0]])
                ->options($this->getCustomGenders($context))
                ->yup(
                    Yup::number()
                        ->nullable()
                        ->when(
                            Yup::when('gender')
                                ->is(0)
                                ->then(
                                    Yup::number()
                                        ->nullable()
                                        ->required(__p('user::validation.custom_gender_field_is_a_required_field'))
                                )
                        )
                ),
        );
    }

    protected function addLocationField(Section $basic, bool $isBasicFieldRequired): void
    {
        if (!Settings::get('user.enable_location', false)) {
            return;
        }

        $countries = Country::buildCountrySearchForm();
        $validation = Yup::string();
        $basic->addFields(
            Builder::choice('country_iso')
                ->label(__p('localize::country.country'))
                ->options($countries)
                ->required($isBasicFieldRequired)
                ->yup(
                    $isBasicFieldRequired
                        ? $validation->required(__p('user::validation.country_is_a_required_field'))
                        : $validation
                ),
            Builder::countryStatePicker('country_state')
                ->label(__p('localize::country.state'))
                ->description(__p('localize::country.state_name'))
                ->searchEndpoint('user/country/state')
                ->searchParams([
                    'country' => ':country_iso',
                ])->showWhen([
                    'neq', 'country_iso', null,
                ]),
        );
    }

    protected function addCityField(Section $basic): void
    {
        if (!Settings::get('user.enable_city', false)) {
            return;
        }

        $basic->addField(
            Builder::countryCity('country_city_code')
                ->label(__p('localize::country.city'))
                ->description(__p('localize::country.city_name'))
                ->searchEndpoint('user/city')
                ->searchParams([
                    'country' => ':country_iso',
                    'state'   => ':country_state',
                ])
        );
    }

    protected function handleEmailFields(Section $basic): void
    {
        $basic->addFields(
            Builder::email('email')
                ->autoComplete('off')
                ->marginNormal()
                ->label(__p('core::phrase.email_address'))
                ->placeholder(__p('core::phrase.email_address'))
                ->returnKeyType('next')
                ->required()
                ->shrink()
                ->validateAction('user.user.exist')
                ->yup(
                    Yup::string()
                        ->required(__p('validation.this_field_is_a_required_field'))
                        ->email(__p('validation.field_must_be_a_valid_email'))
                        ->setError('typeError', __p('validation.email_is_a_required_field')),
                ),
        );

        $this->addEmailConfirmationField($basic);
    }

    protected function addEmailConfirmationField(Section $basic): void
    {
        if (!Settings::get('user.force_user_to_reenter_email', false)) {
            return;
        }

        $basic->addFields(
            Builder::email('reenter_email')
                ->autoComplete('off')
                ->marginNormal()
                ->label(__p('core::phrase.reenter_email_address'))
                ->placeholder(__p('core::phrase.reenter_email_address'))
                ->returnKeyType('next')
                ->required()
                ->yup(Yup::string()
                    ->required()
                    ->format('email')
                    ->setError('required', __p('validation.this_field_is_a_required_field'))
                    ->setError('typeError', __p('validation.reenter_email_is_a_required_field'))
                    ->setError('format', __p('validation.invalid_email_address')))
        );
    }

    protected function handleEmailAndPhoneNumberFields(Section $basic): void
    {
        $basic->addFields(
            Builder::email('email')
                ->autoComplete('off')
                ->marginNormal()
                ->label(__p('user::web.email_or_phone'))
                ->placeholder(__p('user::web.email_or_phone'))
                ->returnKeyType('next')
                ->validateAction('user.user.exist')
                ->required()
                ->shrink()
                ->validateAction('user.user.exist')
                ->yup(
                    Yup::string()
                        ->required(__p('validation.this_field_is_a_required_field'))
                        ->matches(
                            sprintf('%s|%s', MetaFoxConstant::EMAIL_REGEX, MetaFoxConstant::PHONE_NUMBER_REGEX),
                            __p('validation.invalid_email_or_phone')
                        ),
                ),
        );

        $this->addEmailAndPhoneNumberConfirmationField($basic);
    }

    protected function addEmailAndPhoneNumberConfirmationField(Section $basic): void
    {
        if (!Settings::get('user.force_user_to_reenter_email', false)) {
            return;
        }

        $basic->addField(
            Builder::email('reenter_email')
                ->autoComplete('off')
                ->marginNormal()
                ->label(__p('user::phrase.reenter_email_or_phone'))
                ->placeholder(__p('user::phrase.reenter_email_or_phone'))
                ->returnKeyType('next')
                ->required()
                ->yup(
                    Yup::string()
                        ->required(__p('validation.this_field_is_a_required_field'))
                        ->matches(
                            sprintf('%s|%s', MetaFoxConstant::EMAIL_REGEX, MetaFoxConstant::PHONE_NUMBER_REGEX),
                            __p('validation.invalid_email_or_phone')
                        ),
                )
        );
    }

    protected function handleUserNameField(Section $basic): void
    {
        $setting = Settings::get('user.available_name_field_on_sign_up');

        if ($setting != UserSupport::DISPLAY_BOTH && $setting != UserSupport::DISPLAY_USER_NAME) {
            return;
        }

        $basic->addField(
            $this->getUserNameField()->marginNormal()
        );
    }

    protected function handleFullNameField(Section $basic): void
    {
        $setting = Settings::get('user.available_name_field_on_sign_up');

        if ($setting != UserSupport::DISPLAY_BOTH && $setting != UserSupport::DISPLAY_FULL_NAME) {
            return;
        }

        $basic->addField(
            $this->getFullNameField()
                ->marginNormal(),
        );
    }

    protected function addEmailConfirmationFieldFlatten(Section $basic): void
    {
        if (!Settings::get('user.force_user_to_reenter_email', false)) {
            return;
        }

        $basic->addFields(
            Builder::email('reenter_email')
                ->autoComplete('off')
                ->marginNone()
                ->variant('standard-outlined')
                ->label(__p('core::phrase.reenter_email_address'))
                ->placeholder(__p('core::phrase.reenter_email_address'))
                ->returnKeyType('next')
                ->required()
                ->yup(Yup::string()
                    ->required()
                    ->format('email')
                    ->setError('required', __p('validation.this_field_is_a_required_field'))
                    ->setError('typeError', __p('validation.reenter_email_is_a_required_field'))
                    ->setError('format', __p('validation.invalid_email_address')))
        );
    }

    protected function handleEmailFieldsFlatten(Section $basic): void
    {
        $basic->addFields(
            Builder::email('email')
                ->autoComplete('off')
                ->marginNone()
                ->variant('standard-outlined')
                ->label(__p('core::phrase.email_address'))
                ->placeholder(__p('core::phrase.email_address'))
                ->returnKeyType('next')
                ->required()
                ->shrink()
                ->validateAction('user.user.exist')
                ->yup(
                    Yup::string()
                        ->required(__p('validation.this_field_is_a_required_field'))
                        ->email(__p('validation.field_must_be_a_valid_email'))
                        ->setError('typeError', __p('validation.email_is_a_required_field')),
                ),
        );

        $this->addEmailConfirmationFieldFlatten($basic);
    }

    protected function handleEmailAndPhoneNumberFieldsFlatten(Section $basic): void
    {
        $basic->addFields(
            Builder::email('email')
                ->autoComplete('off')
                ->marginNone()
                ->variant('standard-outlined')
                ->label(__p('user::web.email_or_phone'))
                ->placeholder(__p('user::web.email_or_phone'))
                ->returnKeyType('next')
                ->validateAction('user.user.exist')
                ->required()
                ->shrink()
                ->validateAction('user.user.exist')
                ->yup(
                    Yup::string()
                        ->required(__p('validation.this_field_is_a_required_field'))
                        ->matches(
                            sprintf('%s|%s', MetaFoxConstant::EMAIL_REGEX, MetaFoxConstant::PHONE_NUMBER_REGEX),
                            __p('validation.invalid_email_or_phone')
                        ),
                ),
        );

        $this->addEmailAndPhoneNumberConfirmationFieldFlatten($basic);
    }

    protected function addEmailAndPhoneNumberConfirmationFieldFlatten(Section $basic): void
    {
        if (!Settings::get('user.force_user_to_reenter_email', false)) {
            return;
        }

        $basic->addField(
            Builder::email('reenter_email')
                ->autoComplete('off')
                ->marginNone()
                ->variant('standard-outlined')
                ->label(__p('user::phrase.reenter_email_or_phone'))
                ->placeholder(__p('user::phrase.reenter_email_or_phone'))
                ->returnKeyType('next')
                ->required()
                ->yup(
                    Yup::string()
                        ->required(__p('validation.this_field_is_a_required_field'))
                        ->matches(
                            sprintf('%s|%s', MetaFoxConstant::EMAIL_REGEX, MetaFoxConstant::PHONE_NUMBER_REGEX),
                            __p('validation.invalid_email_or_phone')
                        ),
                )
        );
    }

    protected function handleUserNameFieldFlatten(Section $basic): void
    {
        $setting = Settings::get('user.available_name_field_on_sign_up');

        if ($setting != UserSupport::DISPLAY_BOTH && $setting != UserSupport::DISPLAY_USER_NAME) {
            return;
        }

        $basic->addField($this->getUserNameField()
            ->marginNone()
            ->variant('standard-outlined'));
    }

    protected function handleFullNameFieldFlatten(Section $basic): void
    {
        $setting = Settings::get('user.available_name_field_on_sign_up');

        if ($setting != UserSupport::DISPLAY_BOTH && $setting != UserSupport::DISPLAY_FULL_NAME) {
            return;
        }

        $basic->addField(
            $this->getFullNameField()
                ->marginNone()
                ->variant('standard-outlined'),
        );
    }

    protected function getFullNameField(): AbstractField
    {
        $fullNameYup = Yup::string()
            ->required()
            ->setError('required', __p('validation.this_field_is_a_required_field'))
            ->setError('typeError', __p('validation.full_name_is_a_required_field'));

        if (Settings::get('user.validate_full_name', true)) {
            $displayNameRegex = Regex::getRegexSetting('display_name');

            $fullNameYup->maxLength(Settings::get('user.maximum_length_for_full_name'))
                ->minLength(3)
                ->matches($displayNameRegex, __p(Settings::get('regex.display_name_regex_error_message')));
        }

        return Builder::text('full_name')
            ->label(__p('user::phrase.display_name'))
            ->placeholder(__p('user::phrase.display_name'))
            ->returnKeyType('next')
            ->required()
            ->yup($fullNameYup);
    }

    protected function getUserNameField(): AbstractField
    {
        $usernameRegex = Regex::getUsernameRegexSetting();

        return Builder::text('user_name')
            ->label(__p('core::phrase.username'))
            ->placeholder(__p('user::phrase.choose_a_username'))
            ->returnKeyType('next')
            ->autoComplete('off')
            ->required()
            ->yup(
                Yup::string()
                    ->required()
                    ->matches($usernameRegex)
                    ->minLength(Settings::get('user.min_length_for_username', 5))
                    ->maxLength(Settings::get(
                        'user.max_length_for_username',
                        MetaFoxConstant::DEFAULT_MAX_TITLE_LENGTH
                    ))
                    ->setError('required', __p('validation.this_field_is_a_required_field'))
                    ->setError('typeError', __p('validation.user_name_is_a_required_field'))
                    ->setError('matches', __p(Settings::get('regex.user_name_regex_error_message')))
                    ->setError('minLength', __p('validation.field_must_be_at_least_min_length_characters', [
                        'field'     => '${path}',
                        'minLength' => '${min}',
                    ])),
            );
    }

    protected function handleCustomField(Section $basic): void
    {
        /** @var ProfileRepositoryInterface $customRepository */
        $customRepository = resolve(ProfileRepositoryInterface::class);

        $customRepository->loadFieldRegistration($basic, MetaFoxConstant::RESOLUTION_MOBILE);
    }
}
