<?php

namespace MetaFox\User\Http\Resources\v1\User;

use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;
use MetaFox\Captcha\Support\Facades\Captcha;
use MetaFox\Form\AbstractForm;
use MetaFox\Form\Builder;
use MetaFox\Form\GenderTrait;
use MetaFox\Form\Section;
use MetaFox\Platform\Facades\Settings;
use MetaFox\Platform\MetaFoxConstant;
use MetaFox\Platform\MetaFoxFileType;
use MetaFox\Platform\Rules\MetaFoxPasswordFormatRule;
use MetaFox\Profile\Repositories\ProfileRepositoryInterface;
use MetaFox\RegexRule\Support\Facades\Regex;
use MetaFox\User\Support\User as UserSupport;
use MetaFox\Yup\StringShape;
use MetaFox\Yup\Yup;

/**
 * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
 * @preload 1
 */
class UserRegisterForm extends AbstractForm
{
    use GenderTrait;

    protected ?string $code       = null;
    protected ?string $inviteCode = null;

    /**
     * @param Request $request
     * @return void
     * @throws AuthorizationException
     */
    public function boot(Request $request): void
    {
        $this->code = $request->input('code');
        $this->inviteCode = $request->input('invite_code');
    }

    protected function prepare(): void
    {
        $value = ['code' => $this->code];
        if ($this->inviteCode) {
            Arr::set($value, 'invite_code', $this->inviteCode);
        }

        $this->title(__p('user::phrase.create_a_new_account'))
            ->action('/register')
            ->asPost()
            ->submitAction('user/register')
            ->setValue($value);
    }

    protected function getPasswordValidate(): StringShape
    {
        $passwordRule = new MetaFoxPasswordFormatRule();

        return Yup::string()
            ->required()
            ->setError('required', __p('validation.this_field_is_a_required_field'))
            ->setError('typeError', __p('validation.password_is_a_required_field'))
            ->setError('minLength', __p('validation.field_must_be_at_least_min_length_characters', [
                'field'     => '${path}',
                'minLength' => '${min}',
            ]))
            ->matchesArray($passwordRule->getFormRules(), $passwordRule->message());
    }

    public function initialize(): void
    {
        $basic = $this->addBasic();
        $isBasicFieldRequired = Settings::get('user.require_basic_field', false);
        $enableDateOfBirth = Settings::get('user.enable_date_of_birth', false);
        $enableGender = Settings::get('user.enable_gender', false);
        $enableLocation = Settings::get('user.enable_location', false);
        $enableCity = Settings::get('user.enable_city', false);
        $minYear = Settings::get('user.date_of_birth_start', 1900);
        $maxYear = Settings::get('user.date_of_birth_end', Carbon::now()->year);
        $minDate = Carbon::create($minYear);
        $maxDate = Carbon::create($maxYear);
        $minDateString = $minDate ? $minDate->toDateString() : $minYear;
        $maxDateString = $maxDate ? $maxDate->endOfYear()->toDateString() : $maxYear;
        $birthdayMessage = __p('validation.invalid_date_of_birth_between', [
            'date_start' => $minDateString,
            'date_end'   => $maxDateString,
        ]);

        $context = user();

        if (Settings::get('user.force_user_to_upload_on_sign_up', false)) {
            $basic->addField(
                Builder::avatarUpload('user_profile')
                    ->required()
                    ->accept(file_type()->getMimeTypeFromType(MetaFoxFileType::PHOTO_TYPE, false))
                    ->label(__p('user::phrase.profile_image'))
                    ->placeholder(__p('user::phrase.profile_image'))
                    ->description(__p('user::phrase.profile_image_desc'))
                    ->yup(
                        Yup::object()->addProperty(
                            'base64',
                            Yup::string()->required(__p('validation.field_is_a_required_field', [
                                'field' => __p('user::phrase.profile_image'),
                            ]))
                        )
                    )
            );
        }

        Log::channel('profiler')->debug('filds', Builder::getFields());

        $this->handleFullNameField($basic);
        $this->handleUserNameField($basic);
        match (Settings::get('user.enable_phone_number_registration')) {
            true    => $this->handleEmailAndPhoneNumberFields($basic),
            default => $this->handleEmailFields($basic),
        };

        $basic->addFields(
            Builder::password('password')
                ->autoComplete('off')
                ->marginNormal()
                ->label(__p('user::phrase.password'))
                ->placeholder(__p('user::phrase.password'))
                ->returnKeyType('next')
                ->required()
                ->shrink()
                ->minLength(Settings::get('user.minimum_length_for_password', 8))
                ->maxLength(Settings::get('user.maximum_length_for_password', 30))
                ->yup($this->getPasswordValidate()),
        );

        if (Settings::get('user.signup_repeat_password')) {
            $basic->addField(
                Builder::password('password_confirmation')
                    ->autoComplete('off')
                    ->marginNormal()
                    ->label(__p('user::phrase.confirm_password'))
                    ->placeholder(__p('user::phrase.re_enter_password'))
                    ->variant('outlined')
                    ->returnKeyType('next')
                    ->noFeedback(true)
                    ->required()
                    ->minLength(Settings::get('user.minimum_length_for_password', 8))
                    ->maxLength(Settings::get('user.maximum_length_for_password', 30))
                    ->yup(
                        Yup::string()
                            ->required()
                            ->setError('required', __p('validation.this_field_is_a_required_field'))
                            ->setError('typeError', __p('validation.password_is_a_required_field')),
                    )
            );
        }

        if ($enableDateOfBirth) {
            $yupBirthdayField = Yup::date()
                ->nullable(true)
                ->minYear((string)$minYear, $birthdayMessage)
                ->maxYear((string)$maxYear, $birthdayMessage)
                ->setError('typeError', $birthdayMessage);

            if ($isBasicFieldRequired) {
                $yupBirthdayField
                    ->required(__p('user::validation.birthday_is_a_required_field'));
            }

            $basic->addField(
                Builder::birthday('birthday')
                    ->label(__p('user::phrase.birthday'))
                    ->required($isBasicFieldRequired)
                    ->setAttribute('minDate', $minDateString)
                    ->setAttribute('maxDate', $maxDateString)
                    ->yup($yupBirthdayField)
            );
        }

        if ($enableGender) {
            $genderField = Yup::number()
                ->nullable();

            if ($isBasicFieldRequired) {
                $genderField
                    ->required(__p('user::validation.gender_is_a_required_field'));
            }

            $basic->addFields(
                Builder::gender('gender')
                    ->label(__p('user::phrase.i_am'))
                    ->required($isBasicFieldRequired)
                    ->options($this->getDefaultGenders($context))
                    ->yup($genderField),
                Builder::customGenders('custom_gender')
                    ->label(__p('user::phrase.custom_gender'))
                    ->showWhen(['and', ['eq', 'gender', 0]])
                    ->requiredWhen(['and', ['eq', 'gender', 0]])
                    ->options($this->getCustomGenders($context))
                    ->yup(
                        Yup::number()
                            ->nullable()
                            ->when(
                                Yup::when('gender')
                                    ->is(0)
                                    ->then(
                                        Yup::number()
                                            ->nullable()
                                            ->required(__p('user::validation.custom_gender_field_is_a_required_field'))
                                    )
                            )
                    ),
            );
        }

        if ($enableLocation) {
            $yupLocationField = Yup::string()
                ->nullable();

            if ($isBasicFieldRequired) {
                $yupLocationField
                    ->required(__p('user::validation.country_is_a_required_field'));
            }

            $basic->addFields(
                Builder::countryState('country_iso')
                    ->label(__p('localize::country.country'))
                    ->valueType('array')
                    ->setAttribute('countryFieldName', 'country_iso')
                    ->setAttribute('stateFieldName', 'country_state_id')
                    ->required($isBasicFieldRequired)
                    ->yup($yupLocationField),
            );
        }

        if ($enableCity) {
            $basic->addField(Builder::countryCity('country_city_code')
                ->label(__p('localize::country.city'))
                ->description(__p('localize::country.city_name'))
                ->searchEndpoint('user/city')
                ->searchParams([
                    'country' => ':country_iso',
                    'state'   => ':country_state_id',
                ]));
        }

        app('events')->dispatch('user.registration.extra_fields.build', [$basic]);

        $this->handleCustomField($basic);

        if (Settings::get('user.new_user_terms_confirmation')) {
            $basic->addField(
                Builder::checkbox('agree')
                    ->label(__p('user::phrase.agree_field_label'))
                    ->returnKeyType('next')
                    ->required()
                    ->uncheckedValue(false)
                    ->yup(
                        Yup::number()
                            ->required()
                            ->setError('required', __p('validation.agree_field_is_a_required_field'))
                            ->setError('typeError', __p('validation.agree_field_is_a_required_field'))
                    )
            );
        }

        $basic->addField(Captcha::getFormField('user.user_signup'));

        $footer = $this->addFooter([
            'style' => [
                'display'        => 'flex',
                'flexWrap'       => 'wrap',
                'justifyContent' => 'space-between',
            ],
        ]);

        $footer->addFields(
            Builder::submit()
                ->sizeLarge()
                ->marginDense()
                ->label(__p('user::phrase.create_account')),
            Builder::linkButton('already_had_an_account')
                ->link('/login')
                ->variant('link')
                ->sizeMedium()
                ->marginDense()
                ->color('primary')
                ->setAttribute('controlProps', [
                    'textAlign' => 'left',
                ])
                ->showWhen(['eq', 'mediaScreen', 'small'])
                ->label(__p('user::phrase.already_had_an_account')),
        );
    }

    protected function handleEmailFields(Section $basic): void
    {
        $basic->addFields(
            Builder::validateText('email')
                ->autoComplete('off')
                ->marginNormal()
                ->label(__p('core::phrase.email_address'))
                ->placeholder(__p('core::phrase.email_address'))
                ->returnKeyType('next')
                ->required()
                ->shrink()
                ->validateAction('user.user.exist')
                ->yup(
                    Yup::string()
                        ->required(__p('validation.this_field_is_a_required_field'))
                        ->email(__p('validation.field_must_be_a_valid_email'))
                        ->setError('typeError', __p('validation.email_is_a_required_field')),
                ),
        );

        $this->addEmailConfirmationField($basic);
    }

    protected function addEmailConfirmationField(Section $basic): void
    {
        if (!Settings::get('user.force_user_to_reenter_email', false)) {
            return;
        }

        $field = Builder::email('reenter_email')
            ->autoComplete('off')
            ->marginNormal()
            ->label(__p('core::phrase.reenter_email_address'))
            ->placeholder(__p('core::phrase.reenter_email_address'))
            ->returnKeyType('next')
            ->required()
            ->yup(Yup::string()
                ->required()
                ->format('email')
                ->setError('required', __p('validation.this_field_is_a_required_field'))
                ->setError('typeError', __p('validation.reenter_email_is_a_required_field'))
                ->setError('format', __p('validation.invalid_email_address')));

        $basic->addField($field);
    }

    protected function handleEmailAndPhoneNumberFields(Section $basic): void
    {
        $basic->addFields(
            Builder::validateText('email')
                ->autoComplete('off')
                ->marginNormal()
                ->label(__p('user::web.email_or_phone'))
                ->placeholder(__p('user::web.email_or_phone'))
                ->returnKeyType('next')
                ->required()
                ->shrink()
                ->validateAction('user.user.exist')
                ->yup(
                    Yup::string()
                        ->required(__p('validation.this_field_is_a_required_field'))
                        ->matches(
                            sprintf('%s|%s', MetaFoxConstant::EMAIL_REGEX, MetaFoxConstant::PHONE_NUMBER_REGEX),
                            __p('validation.invalid_email_or_phone')
                        ),
                ),
        );

        $this->addEmailAndPhoneNumberConfirmationField($basic);
    }

    protected function addEmailAndPhoneNumberConfirmationField(Section $basic): void
    {
        if (!Settings::get('user.force_user_to_reenter_email', false)) {
            return;
        }

        $basic->addField(
            Builder::email('reenter_email')
                ->autoComplete('off')
                ->marginNormal()
                ->label(__p('user::phrase.reenter_email_or_phone'))
                ->placeholder(__p('user::phrase.reenter_email_or_phone'))
                ->returnKeyType('next')
                ->required()
                ->yup(
                    Yup::string()
                        ->required(__p('validation.this_field_is_a_required_field'))
                        ->matches(
                            sprintf('%s|%s', MetaFoxConstant::EMAIL_REGEX, MetaFoxConstant::PHONE_NUMBER_REGEX),
                            __p('validation.invalid_email_or_phone')
                        ),
                )
        );
    }

    protected function handleUserNameField(Section $basic): void
    {
        $setting = Settings::get('user.available_name_field_on_sign_up');

        if ($setting != UserSupport::DISPLAY_BOTH && $setting != UserSupport::DISPLAY_USER_NAME) {
            return;
        }

        $usernameRegex = Regex::getUsernameRegexSetting();

        $basic->addField(
            Builder::validateText('user_name')
                ->marginNormal()
                ->label(__p('core::phrase.username'))
                ->placeholder(__p('user::phrase.choose_a_username'))
                ->returnKeyType('next')
                ->shrink()
                ->autoComplete('off')
                ->validateAction('user.user.exist')
                ->required()
                ->yup(
                    Yup::string()
                        ->required()
                        ->matches($usernameRegex)
                        ->minLength(Settings::get('user.min_length_for_username', 5))
                        ->maxLength(Settings::get(
                            'user.max_length_for_username',
                            MetaFoxConstant::DEFAULT_MAX_TITLE_LENGTH
                        ))
                        ->setError('required', __p('validation.this_field_is_a_required_field'))
                        ->setError('typeError', __p('validation.user_name_is_a_required_field'))
                        ->setError('matches', __p(Settings::get('regex.user_name_regex_error_message'))),
                )
        );
    }

    protected function handleFullNameField(Section $basic): void
    {
        $setting = Settings::get('user.available_name_field_on_sign_up');

        if ($setting != UserSupport::DISPLAY_BOTH && $setting != UserSupport::DISPLAY_FULL_NAME) {
            return;
        }

        $fullNameYup = Yup::string()
            ->required()
            ->setError('required', __p('validation.this_field_is_a_required_field'))
            ->setError('typeError', __p('validation.full_name_is_a_required_field'));

        if (Settings::get('user.validate_full_name', true)) {
            $displayNameRegex = Regex::getRegexSetting('display_name');

            $fullNameYup->maxLength(Settings::get('user.maximum_length_for_full_name'))
                ->minLength(3)
                ->matches($displayNameRegex, __p(Settings::get('regex.display_name_regex_error_message')));
        }

        $basic->addField(
            Builder::text('full_name')
                ->marginNormal()
                ->label(__p('user::phrase.display_name'))
                ->placeholder(__p('user::phrase.display_name'))
                ->returnKeyType('next')
                ->required()
                ->yup($fullNameYup),
        );
    }

    protected function handleCustomField(Section $basic): void
    {
        /** @var ProfileRepositoryInterface $customRepository */
        $customRepository = resolve(ProfileRepositoryInterface::class);

        $customRepository->loadFieldRegistration($basic, MetaFoxConstant::RESOLUTION_WEB);
    }
}
