<?php

namespace MetaFox\User\Http\Requests\v1\User;

use ArrayObject;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Arr;
use MetaFox\Platform\Rules\AllowInRule;
use MetaFox\Platform\Rules\ExistIfGreaterThanZero;
use MetaFox\Platform\Rules\PaginationLimitRule;
use MetaFox\Platform\Support\Helper\Pagination;
use MetaFox\Profile\Repositories\ProfileRepositoryInterface;
use MetaFox\User\Support\Browse\Scopes\User\SortScope;
use MetaFox\User\Support\Browse\Scopes\User\ViewScope;

/**
 * --------------------------------------------------------------------------
 *  Http request for api version v1
 * --------------------------------------------------------------------------.
 *
 * This class is used by automatic dependency injection:
 *
 * @link \MetaFox\User\Http\Controllers\Api\v1\UserController::index;
 * stub: /packages/requests/api_action_request.stub
 */

/**
 * Class IndexRequest.
 */
class IndexRequest extends FormRequest
{
    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, mixed>
     */
    public function rules(): array
    {
        $rules = new ArrayObject([
            'q'                => ['sometimes', 'nullable', 'string'],
            'view'             => ['sometimes', 'string', new AllowInRule(ViewScope::getAllowView())],
            'gender'           => ['sometimes', 'numeric', new ExistIfGreaterThanZero('exists:user_gender,id')],
            'page'             => ['sometimes', 'numeric', 'min:1'],
            'limit'            => ['sometimes', 'numeric', new PaginationLimitRule()],
            'sort'             => ['sometimes', 'string', new AllowInRule(SortScope::getAllowSort())],
            'sort_type'        => ['sometimes', 'string', new AllowInRule(SortScope::getAllowSortType())],
            'country'          => ['sometimes', 'string', 'min:2'],
            'country_state_id' => ['sometimes', 'nullable', 'string'],
            'city'             => ['sometimes', 'nullable'],
            'city_code'        => ['sometimes', 'nullable', 'string'],
            'is_featured'      => ['sometimes', 'numeric'],
        ]);

        $this->customRepository()->loadFieldSearchRules($rules);

        return $rules->getArrayCopy();
    }

    public function validated($key = null, $default = null)
    {
        $data = parent::validated($key, $default);

        $data    = Arr::add($data, 'view', ViewScope::VIEW_DEFAULT);
        $data    = Arr::add($data, 'q', '');
        $data    = Arr::add($data, 'sort', SortScope::getSortDefault());
        $data    = Arr::add($data, 'sort_type', SortScope::getDefaultSortType(Arr::get($data, 'sort')));
        $data    = Arr::add($data, 'limit', Pagination::DEFAULT_ITEM_PER_PAGE);
        $country = Arr::get($data, 'country');

        if (null === $country) {
            Arr::forget($data, ['country', 'country_state_id']);
        }

        $isFeatured = Arr::get($data, 'is_featured');
        if (!$isFeatured) {
            $data['is_featured'] = null;
        }
        $this->handleCustomProfileFields($data);

        return $data;
    }

    /**
     * @param array<string, mixed> $data
     *
     * @return void
     */
    protected function handleCustomProfileFields(array &$data): void
    {
        $fields            = $this->customRepository()->getFieldSearch();
        $result            = [];

        foreach ($fields as $field) {
            $fieldId   = $field->id;
            $fieldName = $field->field_name;
            $value     = Arr::get($data, $fieldName);

            Arr::forget($data, $fieldName);

            if (null == $value) {
                continue;
            }

            $result[] = [
                'id'    => $fieldId,
                'value' => $value,
            ];
        }

        if (!empty($result)) {
            Arr::set($data, 'custom_fields', $result);
        }
    }

    protected function customRepository(): ProfileRepositoryInterface
    {
        return resolve(ProfileRepositoryInterface::class);
    }
}
