<?php

namespace MetaFox\Profile\Support;

use Illuminate\Support\Arr;
use MetaFox\Authorization\Repositories\Contracts\RoleRepositoryInterface;
use MetaFox\Platform\MetaFoxConstant;
use MetaFox\Platform\UserRole;
use MetaFox\Profile\Contracts\CustomFieldSupportInterface;

class CustomField implements CustomFieldSupportInterface
{
    public const TEXT                  = 'text';
    public const RICH_TEXT_EDITOR      = 'richTextEditor';
    public const RADIO_GROUP           = 'radioGroup';
    public const DROPDOWN              = 'dropdown';
    public const CHOICE                = 'choice';
    public const TEXT_AREA             = 'textArea';
    public const CHECK_BOX             = 'checkbox';
    public const MULTI_CHOICE          = 'multiChoice';
    public const DATE                  = 'date';
    public const SEARCH_TEXT_BOX_FIELD = 'searchTextBox';
    public const SWITCH_FIELD          = 'switch';
    public const TYPE_INT              = 'int';
    public const TYPE_STRING           = 'string';
    public const TYPE_ARRAY            = 'array';

    public function getEditTypeAllowOptions(): array
    {
        return [
            self::MULTI_CHOICE,
            self::CHOICE,
            self::RADIO_GROUP,
        ];
    }

    public function transformVarType(string $type): string
    {
        $array = [
            self::MULTI_CHOICE => self::TYPE_ARRAY,
            self::DROPDOWN     => self::TYPE_INT,
            self::CHOICE       => self::TYPE_INT,
            self::RADIO_GROUP  => self::TYPE_INT,
            self::CHECK_BOX    => self::TYPE_INT,
        ];

        if (!Arr::has($array, $type)) {
            return self::TYPE_STRING;
        }

        return $array[$type];
    }

    public function transformForSearch(string $type, ?string $resolution = null): string
    {
        $array = [
            self::TEXT             => self::SEARCH_TEXT_BOX_FIELD,
            self::TEXT_AREA        => self::SEARCH_TEXT_BOX_FIELD,
            self::RICH_TEXT_EDITOR => self::SEARCH_TEXT_BOX_FIELD,
            self::RADIO_GROUP      => self::DROPDOWN,
            self::CHECK_BOX        => self::SWITCH_FIELD,
            self::MULTI_CHOICE     => self::CHOICE,
        ];

        if ($resolution == MetaFoxConstant::RESOLUTION_ADMIN) {
            unset($array[self::CHECK_BOX]);
        }

        if (!Arr::has($array, $type)) {
            return $type;
        }

        return $array[$type];
    }

    public function transformYupType(string $varType, ?string $editType = null): string
    {
        $typeMap = [
            self::TYPE_INT    => 'number',
            self::TYPE_STRING => [
                self::DATE => 'date',
            ],
        ];

        if (!array_key_exists($varType, $typeMap)) {
            return $varType;
        }

        $type = $typeMap[$varType];

        if (is_string($type)) {
            return $type;
        }

        return $type[$editType] ?? $varType;
    }

    //@TODO refactor this => move to database
    public function allowHtml(string $type): bool
    {
        $allowType = [
            self::RICH_TEXT_EDITOR,
        ];

        if (in_array($type, $allowType)) {
            return true;
        }

        return false;
    }

    public function allowLink(string $type): bool
    {
        $allowType = [
            self::RICH_TEXT_EDITOR,
            self::TEXT_AREA,
        ];

        if (in_array($type, $allowType)) {
            return true;
        }

        return false;
    }

    /**
     * @inheritDoc
     */
    public function transformValueForForm(string $type, mixed &$value): mixed
    {
        switch ($type) {
            case self::MULTI_CHOICE:
                $value = $value ? json_decode($value) : [];
                break;
            case self::RADIO_GROUP:
            case self::CHOICE:
                $value = $value ? (int)$value : null;
                break;
        }

        return $value;
    }

    /**
     * @inheritDoc
     */
    public function transformValueForSection(string $type, mixed &$value, array $options): mixed
    {
        switch ($type) {
            case self::MULTI_CHOICE:
                $valueOptions = [];

                if (!is_array($value)) {
                    $value = json_decode($value);
                }

                if (empty($options) || empty($value)) {
                    return $value;
                }

                foreach ($value as $key) {
                    $valueOptions[] = Arr::get($options, $key);
                }

                $value = implode(' - ', $valueOptions);
                break;
            case self::CHECK_BOX:
                $value = match ((bool)$value) {
                    true    => MetaFoxConstant::EMPTY_STRING,
                    default => null
                };
                break;
            case self::RADIO_GROUP:
            case self::CHOICE:
                $value = Arr::get($options, $value);
                break;
        }

        return $value;
    }

    /**
     * @inheritDoc
     */
    public function transformVarTypeForSearch(string $type): string
    {
        $array = [
            self::MULTI_CHOICE => self::TYPE_INT,
            self::DROPDOWN     => self::TYPE_INT,
            self::CHOICE       => self::TYPE_INT,
            self::RADIO_GROUP  => self::TYPE_INT,
            self::CHECK_BOX    => self::TYPE_INT,
        ];

        if (!Arr::has($array, $type)) {
            return self::TYPE_STRING;
        }

        return $array[$type];
    }

    public function getAllowedRoleOptions(): array
    {
        /**@var RoleRepositoryInterface $roleRepository */
        $roleRepository = resolve(RoleRepositoryInterface::class);
        $roles          = $roleRepository->getRoleOptions();

        $disallowedRoleIds = [UserRole::GUEST_USER, UserRole::BANNED_USER];

        return array_filter($roles, function ($role) use ($disallowedRoleIds) {
            return !in_array($role['value'], $disallowedRoleIds);
        });
    }

    public function getAllowedRole(): array
    {
        return Arr::pluck($this->getAllowedRoleOptions(), 'value');
    }
}
