<?php

namespace MetaFox\Profile\Repositories\Eloquent;

use ArrayObject;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Cache;
use MetaFox\Authorization\Models\Role;
use MetaFox\Authorization\Repositories\Contracts\RoleRepositoryInterface;
use MetaFox\Form\AbstractForm;
use MetaFox\Form\Section as SectionForm;
use MetaFox\Platform\Contracts\User;
use MetaFox\Platform\Repositories\AbstractRepository;
use MetaFox\Profile\Models\Field;
use MetaFox\Profile\Models\Profile;
use MetaFox\Profile\Models\Section;
use MetaFox\Profile\Models\Value;
use MetaFox\Profile\Repositories\ProfileRepositoryInterface;
use MetaFox\Profile\Support\CustomField;
use MetaFox\Profile\Support\Facade\CustomField as CustomFieldFacade;

/**
 * stub: /packages/repositories/eloquent_repository.stub.
 */

/**
 * class ProfileRepository.
 */
class ProfileRepository extends AbstractRepository implements ProfileRepositoryInterface
{
    public function model()
    {
        return Profile::class;
    }

    public function loadEditFields(AbstractForm $form, $user, ?string $resolution = null): void
    {
        /** @var Section[] $sections */
        $sections = Section::query()
            ->where('is_active', 1)
            ->orderBy('ordering', 'asc')
            ->get();

        foreach ($sections as $item) {
            $section = $form->addSection($item->name)
                ->label($item->label)
                ->description($item->description);

            $this->loadFieldInSection($section, $user, $item->id, $resolution);
        }
    }

    private function loadFieldInSection($section, $user, int $id, ?string $resolution = null): void
    {
        $model = new Field();
        $query = $model->newQuery()
            ->where('is_active', 1)
            ->where('section_id', $id);

        $query = $this->applyRoleQuery($query, $user);

        /** @var Field[] $fields */
        $fields = $query->orderBy('ordering')->get();

        foreach ($fields as $field) {
            $section->addField($field->toEditField($resolution));
        }
    }

    public function loadEditRules(ArrayObject $rules, $user): void
    {
        /** @var Field[] $fields */
        $query = Field::query()
            ->where('is_active', '=', 1);

        $query  = $this->applyRoleQuery($query, $user);
        $fields = $query->get();

        foreach ($fields as $field) {
            $rules[$field->field_name] = $field->toRule();
        }
    }

    /**
     * @return string[]
     */
    public function getFieldNames(): array
    {
        $result = [];

        /** @var Field[] $fields */
        $fields = Field::query()->newQuery()
            ->where('is_active', '=', 1)
            ->get();

        foreach ($fields as $field) {
            $result[] = $field->field_name;
        }

        return $result;
    }

    /**
     * @return Collection
     */
    public function getFieldRegistration(): Collection
    {
        /** @var Field[] $fields */
        $fields = Field::query()->newQuery()
            ->where('is_active', '=', 1)
            ->where('is_register', '=', 1)
            ->where(function (Builder $whereQuery) {
                $whereQuery->doesntHave('roles')
                    ->orWhereHas('roles', function (Builder $hasQuery) {
                        $hasQuery->whereNull('role_id');
                    });
            })
            ->get();

        return $fields;
    }

    /**
     * @param SectionForm $section
     * @param string|null $resolution
     * @return void
     */
    public function loadFieldRegistration(SectionForm $section, ?string $resolution = null): void
    {
        /** @var Field[] $fields */
        $fields = $this->getFieldRegistration();

        foreach ($fields as $field) {
            $section->addField($field->toEditField($resolution));
        }
    }

    /**
     * @return Collection
     */
    public function getFieldSearch(): Collection
    {
        return localCacheStore()->rememberForever(
            __METHOD__,
            fn() => Field::query()->newQuery()
                ->where('is_active', '=', 1)
                ->where('is_search', '=', 1)
                ->get()
        );
    }

    /**
     * @param SectionForm $section
     * @param string|null $resolution
     * @return void
     */
    public function loadFieldSearch(SectionForm $section, ?string $resolution = null): void
    {
        /** @var Field[] $fields */
        $fields = $this->getFieldSearch();

        foreach ($fields as $field) {
            $section->addField($field->toSearchField($resolution));
        }
    }

    /**
     * @param ArrayObject $rules
     * @return void
     */
    public function loadFieldRegistrationRules(ArrayObject $rules): void
    {
        /** @var Field[] $fields */
        $fields = $this->getFieldRegistration();

        foreach ($fields as $field) {
            $rules[$field->field_name] = $field->toRule();
        }
    }

    /**
     * @param ArrayObject $rules
     * @return void
     */
    public function loadFieldSearchRules(ArrayObject $rules): void
    {
        /** @var Field[] $fields */
        $fields = $this->getFieldSearch();

        foreach ($fields as $field) {
            $rules[$field->field_name] = $field->toSearchRule();
        }
    }

    /**
     * @return array<string, mixed>
     */
    public function getFieldMaps(): array
    {
        $data = localCacheStore()->rememberForever(
            __METHOD__,
            fn() => Field::query()
                ->where('is_active', 1)
                ->pluck('id', 'field_name')
                ->toArray()
        );

        return is_array($data) ? $data : [];
    }

    /**
     * @param User $user
     * @return array
     */
    public function getStructure(User $user): array
    {
        return Cache::rememberForever(
            __METHOD__ . "({$user->entityId()})",
            function () use ($user) {
                $byIds = [];
                $model = new Field();
                $query = $model->newQuery()
                    ->where('is_active', 1);

                $query = $this->applyRoleQuery($query, $user);

                /** @var Field[] $fields */
                $fields = $query->get();

                foreach ($fields as $field) {
                    $byIds['#' . $field->id] = $field;
                }

                return [
                    'byIds' => $byIds,
                ];
            }
        );
    }

    public function saveValues($user, array $input): void
    {
        foreach ($this->getFieldMaps() as $name => $id) {
            $value = Arr::get($input, $name);

            $item = Value::query()->firstOrNew([
                'user_id'   => $user->id,
                'user_type' => $user->entityType(),
                'field_id'  => $id,
            ], [
                'field_value_text' => $value,
            ]);

            $field = Field::query()->find($id);

            if ($field?->edit_type == CustomField::MULTI_CHOICE) {
                $value = json_encode($value);
            }

            $item->fill([
                'field_value_text' => $value,
            ]);

            $item->saveQuietly();
        }
    }

    public function denormalize($user, bool $forForm = true): array
    {
        $result = [];

        $structure = $this->getStructure($user);

        $customValue = Value::query()->where('user_id', '=', $user->id)
            ->get(['field_id', 'field_value_text', 'ordering']);

        foreach ($customValue as $value) {
            $id    = $value['field_id'];
            $value = $value['field_value_text'];
            $field = Arr::get($structure, 'byIds.#' . $id);

            if (!$field instanceof Field) {
                continue;
            }

            $name = $field->field_name;
            if (!$value) {
                continue;
            }

            if (!$name) {
                continue;
            }
            $options = $field?->options->pluck('label', 'id')->toArray();

            $value = match ($forForm) {
                true  => CustomFieldFacade::transformValueForForm($field->edit_type, $value),
                false => CustomFieldFacade::transformValueForSection($field->edit_type, $value, $options)
            };

            Arr::set($result, $name, $value);
        }

        return $result;
    }

    public function viewSections($user, ArrayObject $output): void
    {
        $custom = $this->denormalize($user, false);

        /** @var Section[] $items */
        $items = Section::query()->where('is_active', 1)
            ->orderBy('ordering', 'asc')
            ->get();

        foreach ($items as $item) {
            $fields = $this->viewFields($item->id, $user, $custom);

            $empty = count(array_values($fields));

            if (!$empty) {
                continue;
            }

            $output[$item->name] = [
                'label'       => $item->label,
                'description' => $item->description,
                'fields'      => $fields,
            ];
        }
    }

    public function viewFields($section, $user, array &$data): array
    {
        $response = [];

        $model = new Field();
        $query = $model->newQuery()
            ->where('section_id', $section)
            ->where('is_active', 1);

        $query = $this->applyRoleQuery($query, $user);
        /** @var Field[] $fields */
        $fields = $query->orderBy('ordering', 'asc')->get();

        foreach ($fields as $field) {
            $value = $data[$field->name] ?? null;
            if (null === $value) {
                continue;
            }

            if ($value && !CustomFieldFacade::allowHtml($field->edit_type)) {
                $value = htmlspecialchars($value);
            }

            if (CustomFieldFacade::allowLink($field->edit_type)) {
                $value = parse_output()->parseUrl($value);
            }

            $response[$field->name] = [
                'label'       => $field->label,
                'description' => $field->description,
                'type'        => $field->edit_type,
                'value'       => $value,
            ];
        }

        return $response;
    }

    private function applyRoleQuery(Builder $query, User $user): Builder
    {
        $contextRole = resolve(RoleRepositoryInterface::class)->roleOf($user);
        if (!$contextRole instanceof Role) {
            return $query;
        }

        return $query->where(function (Builder $whereQuery) use ($contextRole) {
            $whereQuery->doesntHave('roles')
                ->orWhereHas('roles', function (Builder $hasQuery) use ($contextRole) {
                    $hasQuery->where('role_id', '=', $contextRole->entityId());
                });
        });
    }
}
