<?php

namespace MetaFox\Profile\Repositories\Eloquent;

use Illuminate\Contracts\Pagination\Paginator;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Cache;
use MetaFox\Platform\MetaFoxConstant;
use MetaFox\Platform\Repositories\AbstractRepository;
use MetaFox\Platform\Support\Browse\Scopes\SearchScope;
use MetaFox\Profile\Models\Field;
use MetaFox\Profile\Repositories\FieldRepositoryInterface;
use MetaFox\Profile\Repositories\OptionRepositoryInterface;
use MetaFox\Profile\Repositories\ValueRepositoryInterface;
use MetaFox\Profile\Support\Facade\CustomField as CustomFieldFacade;

/**
 * stub: /packages/repositories/eloquent_repository.stub.
 */

/**
 * class FieldRepository.
 *
 * @property Field $model
 * @method   Field getModel()
 * @method   Field find($id, $columns = ['*'])
 */
class FieldRepository extends AbstractRepository implements FieldRepositoryInterface
{
    public function model()
    {
        return Field::class;
    }

    protected function optionsRepository(): OptionRepositoryInterface
    {
        return resolve(OptionRepositoryInterface::class);
    }

    protected function valueRepository(): ValueRepositoryInterface
    {
        return resolve(ValueRepositoryInterface::class);
    }

    /**
     * @param array<string, mixed> $attributes
     *
     * @return Paginator
     */
    public function viewFields(array $attributes): Paginator
    {
        $query = $this->buildQueryViewField($attributes);

        return $query
            ->orderBy('ordering')
            ->paginate($attributes['limit'] ?? 100);
    }

    private function buildQueryViewField(array $attributes)
    {
        $name     = Arr::get($attributes, 'name');
        $required = Arr::get($attributes, 'required');
        $active   = Arr::get($attributes, 'active');
        $roleId   = Arr::get($attributes, 'role_id');

        $query = $this->getModel()->newModelQuery();

        if ($name) {
            $searchScope = new SearchScope($name, ['field_name']);
            $query       = $query->addScope($searchScope);
        }

        if ($roleId) {
            $query->where(function ($innerQuery) use ($roleId) {
                $innerQuery->whereHas('roles', function ($q) use ($roleId) {
                    $q->where('role_id', '=', $roleId);
                });

                $innerQuery->orWhereDoesntHave('roles');
            });
        }

        if (null !== $active) {
            $query->where('is_active', $active);
        }

        if (null !== $required) {
            $query->where('is_required', $required);
        }

        return $query;
    }

    public function getActiveFields(): Collection
    {
        return Cache::rememberForever(
            __METHOD__,
            fn() => $this->getModel()
                ->newModelQuery()
                ->where('is_active', 1)
                ->get()
        );
    }

    public function orderFields(array $orderIds): bool
    {
        $fields = Field::query()
            ->whereIn('id', $orderIds)
            ->get()
            ->keyBy('id');

        if (!$fields->count()) {
            return true;
        }

        $ordering = 1;

        foreach ($orderIds as $orderId) {
            $orderField = $fields->get($orderId);

            if (null === $orderField) {
                continue;
            }

            $orderField->update(['ordering' => $ordering++]);
        }

        return true;
    }

    /**
     * @param array $attributes
     * @return Field
     * @throws \Exception
     */
    public function createField(array $attributes): Field
    {
        $currentOrdering        = $this->getModel()->newQuery()->max('ordering');
        $attributes['ordering'] = ++$currentOrdering;
        $description            = Arr::get($attributes, 'description');
        $label                  = Arr::get($attributes, 'label');

        /** @var Field $field */
        $field = $this->getModel()->newModelInstance();

        $field->fill($attributes);
        $field->setDescriptionAttribute($description);
        $field->setLabelAttribute($label);
        $field->save();

        return $field;
    }

    public function updateField(array $attributes, int $id): Field
    {
        $field    = $this->find($id);
        $options  = Arr::get($attributes, 'options', []);
        $editType = Arr::get($attributes, 'edit_type');

        if (!empty($options)) {
            $this->handleOptionFields($field, $options);
            Arr::forget($attributes, 'options');
        }

        if (
            !in_array($editType, CustomFieldFacade::getEditTypeAllowOptions())
            && in_array($field?->edit_type, CustomFieldFacade::getEditTypeAllowOptions())
        ) {
            $removeIds = $field?->options?->pluck('id')->toArray() ?? [];

            $field?->options()?->delete();
            $this->valueRepository()->deleteValue($field, $removeIds);
        }

        $field->setDescriptionAttribute($attributes['description']);
        $field->setLabelAttribute($attributes['label']);
        $field->update($attributes);
        $field->refresh();

        return $field;
    }

    protected function handleOptionFields(Field $field, array $attributes)
    {
        $newField = array_filter($attributes, function ($item) {
            if (isset($item['status'])) {
                return $item['status'] == MetaFoxConstant::FILE_CREATE_STATUS;
            }

            return false;
        });

        $removedField = array_filter($attributes, function ($item) {
            if (isset($item['status'])) {
                return $item['status'] == MetaFoxConstant::FILE_REMOVE_STATUS;
            }

            return false;
        });

        $updatedField = array_filter($attributes, function ($item) {
            if (isset($item['status'])) {
                return $item['status'] == MetaFoxConstant::FILE_UPDATE_STATUS;
            }

            return false;
        });

        if (!empty($newField)) {
            $this->optionsRepository()->createOptions($field, $newField);
        }

        if (!empty($removedField)) {
            $this->optionsRepository()->removeOptions($field, $removedField);
        }

        if (!empty($updatedField)) {
            $this->optionsRepository()->updateOptions($field, $updatedField);
        }
    }

    public function toggleActive(int $id, array $attributes): Field
    {
        $field = $this->find($id);

        $field->update($attributes);

        return $field;
    }
}
