<?php

namespace MetaFox\Profile\Models;

use Illuminate\Database\Eloquent\Builder as EloquentBuilder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Support\Collection;
use MetaFox\Authorization\Models\Role;
use MetaFox\Form\Builder;
use MetaFox\Form\FormField;
use MetaFox\Form\Mobile\Builder as MobileBuilder;
use MetaFox\Localize\Repositories\PhraseRepositoryInterface;
use MetaFox\Platform\Contracts\Entity;
use MetaFox\Platform\MetaFox;
use MetaFox\Platform\MetaFoxConstant;
use MetaFox\Platform\Traits\Eloquent\Model\HasEntity;
use MetaFox\Platform\Traits\Eloquent\Model\HasNestedAttributes;
use MetaFox\Profile\Database\Factories\FieldFactory;
use MetaFox\Profile\Support\CustomField;
use MetaFox\Profile\Support\Facade\CustomField as CustomFieldFacade;
use MetaFox\Yup\Yup;

/**
 * stub: /packages/models/model.stub.
 */

/**
 * class Field.
 *
 * @mixin EloquentBuilder
 * @property        int        $id
 * @property        string     $section_id
 * @property        string     $field_name
 * @property        string     $module_id
 * @property        string     $product_id
 * @property        ?int       $role_id
 * @property        int        $privacy
 * @property        string     $type_id
 * @property        string     $edit_type
 * @property        string     $view_type
 * @property        string     $var_type
 * @property        bool       $is_active
 * @property        bool       $is_required
 * @property        bool       $is_feed
 * @property        int        $ordering
 * @property        bool       $is_register
 * @property        bool       $is_search
 * @property        bool       $has_description
 * @property        bool       $has_label
 * @property        string     $label
 * @property        string     $editingLabel
 * @property        string     $editingDescription
 * @property        ?Section   $section
 * @property        Collection $options
 * @property        string     $description
 * @property        string     $name
 * @property        ?array     $extra
 * @property        Collection $roles
 * @method   static FieldFactory factory(...$parameters)
 */
class Field extends Model implements Entity
{
    use HasEntity;
    use HasNestedAttributes;
    use HasFactory;

    public const ENTITY_TYPE = 'user_custom_field';

    protected $table = 'user_custom_fields';

    public $timestamps = false;

    /** @var string[] */
    protected $fillable = [
        'section_id',
        'field_name',
        'is_section',
        'type_id',
        'edit_type',
        'view_type',
        'var_type',
        'privacy',
        'ordering',
        'is_active',
        'is_required',
        'is_feed',
        'is_register',
        'is_search',
        'has_label',
        'label',
        'description',
        'has_description',
        'extra',
    ];

    protected $casts = [
        'is_active'   => 'boolean',
        'is_register' => 'boolean',
        'is_required' => 'boolean',
        'extra'       => 'array',
    ];
    /**
     * @var array<string>|array<string, mixed>
     */
    public array $nestedAttributes = [
        'options',
        'roles',
    ];

    /**
     * @return FieldFactory
     */
    protected static function newFactory()
    {
        return FieldFactory::new();
    }

    /**
     * @return BelongsToMany
     */
    public function roles(): BelongsToMany
    {
        return $this->belongsToMany(
            Role::class,
            'user_custom_field_role_data',
            'field_id',
            'role_id'
        )->using(FieldRoleData::class);
    }

    public function getLabelAttribute(): ?string
    {
        if (!$this->has_label) {
            return null;
        }

        return __p('profile::phrase.' . $this->field_name . '_label');
    }

    public function getEditingLabelAttribute()
    {
        return __p('profile::phrase.' . $this->field_name . '_label');
    }

    public function getNameAttribute()
    {
        return $this->field_name;
    }

    public function setLabelAttribute($value)
    {
        $key = 'profile::phrase.' . $this->field_name . '_label';

        if (!is_array($value)) {
            return;
        }

        $service = resolve(PhraseRepositoryInterface::class);

        foreach ($value as $locale => $text) {
            $service->updatePhraseByKey($key, strip_tags($text), $locale);
        }
    }

    public function setDescriptionAttribute($value)
    {
        $key = 'profile::phrase.' . $this->field_name . '_description';

        if (!is_array($value)) {
            return;
        }

        $service = resolve(PhraseRepositoryInterface::class);

        foreach ($value as $locale => $text) {
            $service->updatePhraseByKey($key, strip_tags($text), $locale);
        }
    }

    public function getDescriptionAttribute(): ?string
    {
        if (!$this->has_description) {
            return null;
        }

        return __p('profile::phrase.' . $this->field_name . '_description');
    }

    public function getEditingDescriptionAttribute(): ?string
    {
        if (!$this->has_description) {
            return null;
        }

        return __p('profile::phrase.' . $this->field_name . '_description');
    }

    private function getCreator(?string $resolution = null, bool $allowSearch = false): ?string
    {
        $editType = $this->edit_type;
        if ($allowSearch) {
            $editType = CustomFieldFacade::transformForSearch($this->edit_type, $resolution);
        }

        return match ($resolution) {
            MetaFoxConstant::RESOLUTION_MOBILE => MobileBuilder::getCreator($this->edit_type),
            default                            => Builder::getCreator($editType),
        };
    }

    private function getOptions(mixed &$field): void
    {
        if (method_exists($field, 'options')) {
            // put option to fields.
            $options = $this->options->map(function (Option $item) {
                return ['label' => $item->label, 'value' => $item->entityId()];
            })->toArray();

            $field->options($options);
        }
    }

    public function toEditField(?string $resolution = null): ?FormField
    {
        $creator = $this->getCreator($resolution);

        if (!$creator) {
            return null;
        }

        $data = [
            'name'        => $this->field_name,
            'label'       => $this->editingLabel,
            'description' => strip_tags($this->editingDescription),
            'required'    => $this->is_required,
        ];

        $field = new $creator($data);

        switch ($this->edit_type) {
            case CustomField::RADIO_GROUP:
                if (!MetaFox::isMobile()) {
                    $field->setAttribute('descriptionPlacement', 'bottom');
                }
                break;
            case CustomField::CHECK_BOX:
                $field->setAttribute('uncheckedValue', false);
                break;
        }

        $this->getOptions($field);
        $this->setYupValidation($field);

        return $field;
    }

    protected function setYupValidation(mixed &$field): void
    {
        $yupType = CustomFieldFacade::transformYupType($this->var_type, $this->edit_type);

        $yup = match ($this->is_required) {
            true  => Yup::$yupType()->required(),
            false => Yup::$yupType()->nullable()
        };

        $this->setValidationRules($yup);

        $field->yup($yup);
    }

    protected function setValidationRules(mixed $yup): void
    {
        switch ($this->edit_type) {
            case CustomField::DATE:
                $yup->setError('typeError', __p('core::phrase.invalid_date'));
                break;

            case CustomField::CHECK_BOX:
            case CustomField::MULTI_CHOICE:
                if ($this->is_required) {
                    $yup->setError('typeError', __p('validation.this_field_is_a_required_field'));
                }

                break;
        }
    }

    public function toSearchField(?string $resolution = null): ?FormField
    {
        $creator = $this->getCreator($resolution, true);

        if (!$creator) {
            return null;
        }

        $field = new $creator([
            'name'        => $this->field_name,
            'label'       => $this->editingLabel,
            'description' => $this->editingDescription,
        ]);

        $this->getOptions($field);

        if (MetaFox::isMobile()) {
            return $field->marginNone();
        }

        return $field->marginDense();
    }

    public function section(): ?HasOne
    {
        return $this->hasOne(Section::class, 'id', 'section_id');
    }

    public function options(): HasMany
    {
        return $this->hasMany(Option::class, 'field_id', 'id');
    }

    public function toRule(): array
    {
        $this->var_type = CustomFieldFacade::transformVarType($this->edit_type);

        if ($this->is_required) {
            return ['required', $this->var_type];
        }

        return [$this->var_type, 'sometimes', 'nullable'];
    }

    public function toSearchRule(): array
    {
        $this->var_type = CustomFieldFacade::transformVarTypeForSearch($this->edit_type);

        return [$this->var_type, 'sometimes', 'nullable'];
    }
}

// end
