<?php

namespace MetaFox\Profile\Http\Resources\v1\Field\Admin;

use MetaFox\Form\AbstractField;
use MetaFox\Form\AbstractForm;
use MetaFox\Form\Builder as Builder;
use MetaFox\Platform\MetaFoxConstant;
use MetaFox\Profile\Models\Field as Model;
use MetaFox\Profile\Models\Section;
use MetaFox\Profile\Support\CustomField as CustomFieldSupport;
use MetaFox\Profile\Support\Facade\CustomField as CustomFieldFacade;
use MetaFox\Yup\Yup;

/**
 * --------------------------------------------------------------------------
 * Form Configuration
 * --------------------------------------------------------------------------
 * stub: /packages/resources/edit_form.stub.
 */

/**
 * Class CreateFieldForm.
 * @property ?Model $resource
 * @ignore
 * @codeCoverageIgnore
 */
class CreateFieldForm extends AbstractForm
{
    public const MAX_NAME_LENGTH = 32;

    protected function prepare(): void
    {
        $this->title(__p('core::phrase.edit'))
            ->action(apiUrl('admin.profile.field.store'))
            ->asPost()
            ->setValue([
                'section_id'      => 1,
                'type_id'         => 'main',
                'edit_type'       => CustomFieldSupport::TEXT,
                'view_type'       => 'text',
                'var_type'        => CustomFieldSupport::TYPE_STRING,
                'has_label'       => 1,
                'has_description' => 1,
                'is_active'       => 1,
            ]);
    }

    protected function initialize(): void
    {
        $basic = $this->addBasic();
        $basic->addFields(
            Builder::text('field_name')
                ->maxLength(self::MAX_NAME_LENGTH)
                ->label(__p('core::phrase.name'))
                ->description(__p('profile::phrase.name_desc'))
                ->required()
                ->yup(
                    Yup::string()
                        ->required()
                        ->matches(MetaFoxConstant::RESOURCE_IDENTIFIER_REGEX, __p('validation.alpha_underscore_lower_only', [
                            'attribute' => '${path}',
                        ]))
                ),
            Builder::translatableText('label')
                ->label(__p('core::phrase.label'))
                ->required()
                ->buildFields(),
            Builder::translatableText('description')
                ->asTextArea()
                ->label(__p('core::phrase.description'))
                ->buildFields(),
            Builder::dropdown('section_id')
                ->required()
                ->label(__p('profile::phrase.group'))
                ->options($this->getLocationOptions())
                ->yup(Yup::number()->required()),
            Builder::dropdown('edit_type')
                ->label(__p('profile::phrase.edit_type_label'))
                ->options($this->getEditTypeOptions()),
            Builder::customField('options')
                ->showWhen(['includes', 'edit_type', CustomFieldFacade::getEditTypeAllowOptions()])
                ->minLength(1),
            Builder::choice('roles')
                ->multiple($this->canAssignMultipleRoles())
                ->label(__p('core::phrase.role'))
                ->options(CustomFieldFacade::getAllowedRoleOptions()),
            Builder::checkbox('has_label')
                ->label(__p('profile::phrase.has_label')),
            Builder::checkbox('has_description')
                ->label(__p('profile::phrase.has_description')),
            $this->getActiveField(),
            $this->getRegisterField(),
            Builder::checkbox('is_search')
                ->label(__p('profile::phrase.include_on_search_user')),
            Builder::checkbox('is_required')
                ->label(__p('profile::phrase.is_required')),
        );

        $this->addDefaultFooter();
    }

    /**
     * getLocationOptions.
     *
     * @return array<mixed>
     */
    protected function getLocationOptions(): array
    {
        $response = [];

        /** @var Section[] $fields */
        $fields = Section::query()->orderBy('name')->get();

        foreach ($fields as $field) {
            $response[] = ['label' => $field->label, 'value' => $field->id];
        }

        return $response;
    }

    /**
     * getEditTypeOptions.
     *
     * @return array<mixed>
     */
    protected function getEditTypeOptions(): array
    {
        return [
            [
                'value' => CustomFieldSupport::TEXT,
                'label' => __p('profile::phrase.edit_type.text'),
            ],
            [
                'value' => CustomFieldSupport::TEXT_AREA,
                'label' => __p('profile::phrase.edit_type.textarea'),
            ],
            [
                'value' => CustomFieldSupport::RICH_TEXT_EDITOR,
                'label' => __p('profile::phrase.edit_type.rich_text_editor'),
            ],
            [
                'value' => CustomFieldSupport::CHOICE,
                'label' => __p('profile::phrase.edit_type.selection'),
            ],
            [
                'value' => CustomFieldSupport::CHECK_BOX,
                'label' => __p('profile::phrase.edit_type.checkbox'),
            ],
            [
                'value' => CustomFieldSupport::MULTI_CHOICE,
                'label' => __p('profile::phrase.edit_type.multiple_selection'),
            ],
            [
                'value' => CustomFieldSupport::RADIO_GROUP,
                'label' => __p('profile::phrase.edit_type.radio'),
            ],
            [
                'value' => CustomFieldSupport::DATE,
                'label' => __p('profile::phrase.edit_type.date'),
            ],
        ];
    }

    public function getActiveField(): ?AbstractField
    {
        return Builder::checkbox('is_active')
            ->label(__p('core::phrase.is_active'));
    }

    public function getRegisterField(): ?AbstractField
    {
        return Builder::checkbox('is_register')
            ->label(__p('profile::phrase.is_register'));
    }

    public static function canAssignMultipleRoles(): bool
    {
        return false;
    }
}
