<?php

namespace MetaFox\Profile\Http\Requests\v1\Field\Admin;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Arr;
use MetaFox\Core\Support\Facades\Language;
use MetaFox\Localize\Rules\TranslatableTextRule;
use MetaFox\Platform\MetaFoxConstant;
use MetaFox\Platform\Rules\AllowInRule;
use MetaFox\Platform\Rules\CaseInsensitiveUnique;
use MetaFox\Profile\Http\Resources\v1\Field\Admin\CreateFieldForm;
use MetaFox\Profile\Rules\OptionsFieldRule;
use MetaFox\Profile\Support\Facade\CustomField as CustomFieldFacade;

/**
 * --------------------------------------------------------------------------
 *  Http request for api version v1
 * --------------------------------------------------------------------------.
 *
 * This class is used by automatic dependency injection:
 *
 * @link \MetaFox\Profile\Http\Controllers\Api\v1\FieldAdminController::store
 * stub: /packages/requests/api_action_request.stub
 */

/**
 * Class StoreRequest.
 */
class StoreRequest extends FormRequest
{
    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, mixed>
     */
    public function rules()
    {
        $rules = [
            'section_id'       => ['int', 'sometimes', 'nullable'],
            'field_name'       => [
                'string', 'required', 'regex:/' . MetaFoxConstant::RESOURCE_IDENTIFIER_REGEX . '/',
                new CaseInsensitiveUnique('user_custom_fields', 'field_name'),
                'max:' . CreateFieldForm::MAX_NAME_LENGTH,
            ],
            'type_id'          => ['string', 'required'],
            'edit_type'        => ['string', 'sometimes'],
            'options'          => ['array', 'sometimes'],
            'options.*.label'  => ['sometimes', 'max:150', new OptionsFieldRule()],
            'options.*.status' => [
                'string', new AllowInRule([
                    MetaFoxConstant::FILE_CREATE_STATUS,
                    MetaFoxConstant::FILE_REMOVE_STATUS,
                    MetaFoxConstant::FILE_UPDATE_STATUS,
                ]),
            ],
            'view_type'        => ['string', 'required'],
            'var_type'         => ['string', 'required'],
            'is_active'        => ['sometimes', new AllowInRule([0, 1])],
            'is_required'      => ['sometimes', new AllowInRule([0, 1])],
            'is_register'      => ['sometimes', new AllowInRule([0, 1])],
            'is_search'        => ['sometimes', new AllowInRule([0, 1])],
            'is_feed'          => ['sometimes', new AllowInRule([0, 1])],
            'label'            => ['array', 'required', new TranslatableTextRule()],
            'description'      => ['array', 'sometimes', new TranslatableTextRule(true)],
            'has_label'        => ['int', 'sometimes'],
            'has_description'  => ['int', 'sometimes'],
            'extra'            => ['array', 'sometimes', 'nullable'],
        ];

        return array_merge($rules, $this->fieldRolesRule());
    }

    public function validated($key = null, $default = null)
    {
        $data = parent::validated($key, $default);

        if (isset($data['edit_type'])) {
            Arr::set($data, 'var_type', CustomFieldFacade::transformVarType($data['edit_type']));
            if (!in_array($data['edit_type'], CustomFieldFacade::getEditTypeAllowOptions())) {
                Arr::forget($data, 'options');
            }
        }

        $emptyPhrasesData = Language::getEmptyPhraseData();
        $labelData        = Language::extractPhraseData('label', $data);
        $descriptionData  = Language::extractPhraseData('description', $data);
        $data             = $this->handleFieldRoles($data);

        Arr::set($data, 'label', !empty($labelData) ? $labelData : $emptyPhrasesData);
        Arr::set($data, 'description', !empty($descriptionData) ? $descriptionData : $emptyPhrasesData);

        return $data;
    }

    public function messages(): array
    {
        return [
        ];
    }

    protected function fieldRolesRule(): array
    {
        if (CreateFieldForm::canAssignMultipleRoles()) {
            return [
                'roles'   => ['sometimes', 'array', 'nullable'],
                'roles.*' => ['sometimes', 'integer', 'nullable', new AllowInRule(CustomFieldFacade::getAllowedRole())],
            ];
        }

        return [
            'roles' => ['sometimes', 'integer', 'nullable', new AllowInRule(CustomFieldFacade::getAllowedRole())],
        ];
    }

    protected function handleFieldRoles(array $data): array
    {
        if (!Arr::has($data, 'roles')) {
            return $data;
        }

        $roles = Arr::get($data, 'roles', []);
        if (is_array($roles)) {
            return $data;
        }

        if ($roles) {
            Arr::set($data, 'roles', [$roles]);
            return $data;
        }

        Arr::set($data, 'roles', []);

        return $data;
    }
}
