<?php

namespace MetaFox\EMoney\Support;

use MetaFox\EMoney\Contracts\SupportInterface;
use MetaFox\Platform\Facades\Settings;
use MetaFox\Platform\Contracts\User;
use MetaFox\Platform\MetaFox;
use MetaFox\Platform\MetaFoxConstant;
use MetaFox\User\Repositories\Contracts\UserRepositoryInterface;

class Support implements SupportInterface
{
    public const SERVICE_VISA = 'visa';

    public const TEST_MODE = 'test';
    public const LIVE_MODE = 'live';

    public const TARGET_EXCHANGE_RATE_TYPE_AUTO    = 'auto';
    public const TARGET_EXCHANGE_RATE_TYPE_MANUAL  = 'manual';
    public const DEFAULT_TARGET_EXCHANGE_RATE_TYPE = self::TARGET_EXCHANGE_RATE_TYPE_AUTO;

    public const DEFAULT_TARGET_CURRENCY_CONVERSION_RATE = 'USD';

    public const MAXIMUM_EXCHANGE_RATE_DECIMAL_PLACE_NUMBER = 10;
    public const MINIMUM_EXCHANGE_RATE_NUMBER               = 0.0000000001;

    public const TRANSACTION_STATUS_APPROVED = 'approved';
    public const TRANSACTION_STATUS_PENDING  = 'pending';

    public const WITHDRAW_STATUS_WAITING_CONFIRMATION = 'waiting_confirmation';
    public const WITHDRAW_STATUS_PENDING              = 'pending';
    public const WITHDRAW_STATUS_PROCESSING           = 'processing';
    public const WITHDRAW_STATUS_PROCESSED            = 'processed';
    public const WITHDRAW_STATUS_CANCELLED            = 'cancelled';
    public const WITHDRAW_STATUS_DENIED               = 'denied';

    public const MINIMUM_WITHDRAW_AMOUNT = 0.01;

    public const WITHDRAW_REQUEST_REASON_TYPE_CANCEL  = 'cancel';
    public const WITHDRAW_REQUEST_REASON_TYPE_DENY    = 'deny';

    /**
     * Received money when someone bought your listing
     */
    public const INCOMING_TRANSACTION_TYPE_RECEIVED = 'received';

    /**
     * Using for purchased like a Payment Gateway
     */
    public const OUTGOING_TRANSACTION_TYPE_PURCHASED = 'purchased';
    public const OUTGOING_TRANSACTION_TYPE_WITHDRAWN = 'withdrawn';

    public const TRANSACTION_SOURCE_INCOMING = 'incoming';
    public const TRANSACTION_SOURCE_OUTGOING = 'outgoing';

    /**
     * Apply firstly for USD.
     * @return float
     */
    public function getMinimumWithdrawalAmount(): float
    {
        $min = (float) Settings::get('ewallet.minimum_withdraw', 0);

        if ($min > 0) {
            return $min;
        }

        return self::MINIMUM_WITHDRAW_AMOUNT;
    }

    public function getDefaultCurrency(): string
    {
        return self::DEFAULT_TARGET_CURRENCY_CONVERSION_RATE;
    }

    public function getNotifiable(): ?User
    {
        return resolve(UserRepositoryInterface::class)->getSuperAdmin();
    }

    public function getRequestStatusOptions(): array
    {
        return [
            [
                'label' => __p('core::phrase.pending'),
                'value' => self::WITHDRAW_STATUS_PENDING,
            ],
            [
                'label' => __p('ewallet::phrase.processing'),
                'value' => self::WITHDRAW_STATUS_PROCESSING,
            ],
            [
                'label' => __p('ewallet::phrase.completed'),
                'value' => self::WITHDRAW_STATUS_PROCESSED,
            ],
            [
                'label' => __p('ewallet::phrase.cancelled'),
                'value' => self::WITHDRAW_STATUS_CANCELLED,
            ],
            [
                'label' => __p('ewallet::phrase.denied'),
                'value' => self::WITHDRAW_STATUS_DENIED,
            ],
        ];
    }

    public function getTransactionStatusOptions(): array
    {
        return [
            [
                'label' => __p('core::phrase.pending'),
                'value' => self::TRANSACTION_STATUS_PENDING,
            ],
            [
                'label' => __p('ewallet::phrase.approved'),
                'value' => self::TRANSACTION_STATUS_APPROVED,
            ],
        ];
    }

    public function getBaseCurrencyOptions(?string $target = null): array
    {
        $currencies = app('currency')->getCurrencies();

        if ($target) {
            $currencies = array_filter($currencies, function ($currency) use ($target) {
                return $currency['code'] != $target;
            });
        }

        if (!count($currencies)) {
            return [];
        }

        return array_values(array_map(function ($currency) {
            return [
                'label' => $currency['code'],
                'value' => $currency['code'],
            ];
        }, $currencies));
    }

    public function getRequestStatuses(): array
    {
        return array_column($this->getRequestStatusOptions(), 'value');
    }

    public function isUsingNewAlias(): bool
    {
        if (!MetaFox::isMobile()) {
            return true;
        }

        if (version_compare(MetaFox::getApiVersion(), 'v1.6', '>=')) {
            return true;
        }

        return false;
    }

    public function getAppAlias(): string
    {
        if ($this->isUsingNewAlias()) {
            return 'ewallet';
        }

        return 'emoney';
    }

    public function getSourceOptions(): array
    {
        return [
            [
                'label' => __p('ewallet::phrase.incoming'),
                'value' => self::TRANSACTION_SOURCE_INCOMING,
            ],
            [
                'label' => __p('ewallet::phrase.outgoing'),
                'value' => self::TRANSACTION_SOURCE_OUTGOING,
            ],
        ];
    }

    public function getTypeOptions(): array
    {
        return [
            [
                'label' => __p('ewallet::phrase.bought_your_item'),
                'value' => self::INCOMING_TRANSACTION_TYPE_RECEIVED,
            ],
            [
                'label' => __p('ewallet::phrase.purchased_an_item'),
                'value' => self::OUTGOING_TRANSACTION_TYPE_PURCHASED,
            ],
            [
                'label' => __p('ewallet::phrase.withdrawn'),
                'value' => self::OUTGOING_TRANSACTION_TYPE_WITHDRAWN,
            ],
        ];
    }
}
