<?php
namespace MetaFox\EMoney\Support;

use MetaFox\EMoney\Contracts\PaymentInterface;
use MetaFox\EMoney\Facades\Payment as EwalletPayment;
use MetaFox\EMoney\Repositories\StatisticRepositoryInterface;
use MetaFox\EMoney\Repositories\TransactionRepositoryInterface;
use MetaFox\EMoney\Services\Contracts\ConversionRateServiceInterface;
use MetaFox\Payment\Models\Order;

class PaymentSupport implements PaymentInterface
{

    public function processPayment(Order $order): ?array
    {
        if (null === $order->user) {
            return null;
        }

        $amount = $this->convertAmount($order->total, $order->currency);

        if (null === $amount) {
            return null;
        }

        $balance = resolve(StatisticRepositoryInterface::class)->getUserBalance($order->user, Support::DEFAULT_TARGET_CURRENCY_CONVERSION_RATE);

        if ($balance < $amount) {
            return null;
        }

        $gatewayOrderId = EwalletPayment::generateOrderId($order);

        resolve(TransactionRepositoryInterface::class)->createOutgoingTransaction($order, $gatewayOrderId);

        return [
            'gateway_order_id' => $gatewayOrderId,
        ];
    }

    public function generateTransactionId(Order $order): string
    {
        return sprintf('et_%s', md5(uniqid() . $order->entityId() . $order->entityType()));
    }

    public function generateOrderId(Order $order): string
    {
        return sprintf('eo_%s', md5(uniqid() . $order->entityId() . $order->entityType()));
    }

    private function convertAmount(float $amount, string $baseCurrency, string $targetCurrency = Support::DEFAULT_TARGET_CURRENCY_CONVERSION_RATE): ?float
    {
        if ($baseCurrency == $targetCurrency) {
            return $amount;
        }

        return resolve(ConversionRateServiceInterface::class)->getConversedAmount($baseCurrency, $amount, $targetCurrency);
    }
}
