<?php
namespace MetaFox\EMoney\Support\Gateway;

use Illuminate\Support\Arr;
use MetaFox\EMoney\Facades\Payment as EwalletPayment;
use MetaFox\EMoney\Models\Transaction;
use MetaFox\EMoney\Repositories\StatisticRepositoryInterface;
use MetaFox\EMoney\Support\Support;
use MetaFox\Payment\Models\Order;
use MetaFox\Payment\Support\AbstractPaymentGateway;
use MetaFox\Payment\Support\Facades\Payment;
use MetaFox\Platform\Contracts\User;
use MetaFox\Platform\MetaFox;

class EwalletPaymentGateway extends AbstractPaymentGateway
{
    public const GATEWAY_SERVICE_NAME = 'ewallet';

    public static function getGatewayServiceName(): string
    {
        return self::GATEWAY_SERVICE_NAME;
    }

    public function createGatewayOrder(Order $order, array $params = []): array
    {
        $data = $order->toGatewayOrder();

        if (!$data) {
            throw new \RuntimeException('Invalid order.');
        }

        $result = EwalletPayment::processPayment($order);

        $valid = is_array($result) && Arr::has($result, 'gateway_order_id');

        $redirectUrl = Arr::get($params, 'return_url');

        if (!$valid) {
            $redirectUrl = Arr::get($params, 'cancel_url');
        }

        // Return data based on result of previous step
        return [
            'status'               => $valid,
            'gateway_order_id'     => Arr::get($result, 'gateway_order_id'),
            'gateway_redirect_url' => $this->toRedirectUrl($redirectUrl),
        ];
    }

    protected function toRedirectUrl(?string $url = null): ?string
    {
        if (null === $url) {
            return null;
        }

        if (MetaFox::isMobile()) {
            return url_utility()->convertUrlToLink($url, true);
        }

        return $url;
    }

    public function getGatewayTransaction(string $gatewayTransactionId): ?array
    {
        return null;
    }

    public function getGatewayOrder(string $gatewayOrderId): ?array
    {
        return null;
    }

    public function hasAccess(User $context, array $params): bool
    {
        if (!parent::hasAccess($context, $params)) {
            return false;
        }

        $price = Arr::get($params, 'price');

        if (!is_numeric($price)) {
            return false;
        }

        $price = (float) $price;

        if ($price <= 0) {
            return false;
        }

        $currency = app('currency')->getUserCurrencyId($context);

        $balanceCurrency = Support::DEFAULT_TARGET_CURRENCY_CONVERSION_RATE;

        $balance = resolve(StatisticRepositoryInterface::class)->getUserBalance($context, $balanceCurrency);

        if ($balance <= 0) {
            return false;
        }

        if ($currency != $balanceCurrency) {
            $price = app('ewallet.conversion-rate')->getConversedAmount($currency, $price, $balanceCurrency);
        }

        /**
         * Conversion Rate is not available, so price will be null
         */
        if (null === $price) {
            return false;
        }

        if ($balance < $price) {
            return false;
        }

        return true;
    }
}
