<?php

namespace MetaFox\EMoney\Repositories\Eloquent;

use Illuminate\Contracts\Pagination\Paginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Arr;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Notification;
use MetaFox\EMoney\Facades\Emoney;
use MetaFox\EMoney\Facades\Payment;
use MetaFox\EMoney\Notifications\ApprovedTransactionNotification;
use MetaFox\EMoney\Repositories\StatisticRepositoryInterface;
use MetaFox\EMoney\Services\Contracts\ConversionRateServiceInterface;
use MetaFox\EMoney\Support\Browse\Scopes\GeneralScope;
use MetaFox\EMoney\Support\Support;
use MetaFox\Payment\Models\Order;
use MetaFox\Payment\Models\Transaction as PaymentTransaction;
use MetaFox\Platform\Contracts\Entity;
use MetaFox\Platform\Contracts\User;
use MetaFox\Platform\Facades\Settings;
use MetaFox\Platform\MetaFoxConstant;
use MetaFox\Platform\PackageManager;
use MetaFox\Platform\Repositories\AbstractRepository;
use MetaFox\EMoney\Repositories\TransactionRepositoryInterface;
use MetaFox\EMoney\Models\Transaction;
use MetaFox\Platform\Support\Helper\Pagination;

/**
 * stub: /packages/repositories/eloquent_repository.stub.
 */

/**
 * Class TransactionRepository.
 */
class TransactionRepository extends AbstractRepository implements TransactionRepositoryInterface
{
    public function model()
    {
        return Transaction::class;
    }

    public function createTransactionForIntegration(User $user, User $owner, Entity $entity, string $currency, float $total, ?float $commissionPercentage = null, ?int $holdingDays = null): ?Transaction
    {
        return $this->createModel($user, $owner, $entity, $currency, $total, $commissionPercentage, $holdingDays);
    }

    public function createTrackingTransaction(User $user, User $owner, Entity $entity, string $currency, float $total, string $source = Support::TRANSACTION_SOURCE_OUTGOING, string $type = Support::OUTGOING_TRANSACTION_TYPE_WITHDRAWN): Transaction
    {
        $currentBalance = app('ewallet.statistic')->getUserBalance($owner, $currency);

        $attributes = [
            'user_id'             => $user->entityId(),
            'user_type'           => $user->entityType(),
            'owner_id'            => $owner->entityId(),
            'owner_type'          => $owner->entityType(),
            'item_id'             => $entity->entityId(),
            'item_type'           => $entity->entityType(),
            'module_id'           => PackageManager::getAliasForEntityType($entity->entityType()),
            'total_currency'      => $currency,
            'total_price'         => $total,
            'commission_currency' => $currency,
            'commission_price'    => 0,
            'actual_currency'     => $currency,
            'actual_price'        => $total,
            'balance_price'       => $total,
            'balance_currency'    => $currency,
            'current_balance_price' => $currentBalance,
            'exchange_rate'       => 0,
            'available_at'        => Carbon::now(),
            'status'              => Support::TRANSACTION_STATUS_APPROVED,
            'source'              => $source,
            'type'                => $type,
        ];

        /**
         * @var Transaction $model
         */
        $model = $this->getModel()->newInstance($attributes);

        $model->save();

        return $model->refresh();
    }

    private function createModel(
        User $user,
        User $owner,
        Entity $entity,
        string $currency,
        float $total,
        ?float $commissionPercentage = null,
        ?int $holdingDays = null,
        string $source = Support::TRANSACTION_SOURCE_INCOMING,
        string $type = Support::INCOMING_TRANSACTION_TYPE_RECEIVED,
        ?string $outgoingOrderId = null,
        ?array $extra = null): ?Transaction
    {
        $commission  = $this->getRateService()->getCommissionFee($total, $commissionPercentage);

        $actual      = $total - $commission;

        if (null === $holdingDays) {
            $holdingDays = (int) Settings::get('ewallet.balance_holding_duration', 0);
        }

        $attributes = [
            'user_id'             => $user->entityId(),
            'user_type'           => $user->entityType(),
            'owner_id'            => $owner->entityId(),
            'owner_type'          => $owner->entityType(),
            'item_id'             => $entity->entityId(),
            'item_type'           => $entity->entityType(),
            'module_id'           => PackageManager::getAliasForEntityType($entity->entityType()),
            'total_currency'      => $currency,
            'total_price'         => $total,
            'commission_currency' => $currency,
            'commission_price'    => $commission,
            'actual_currency'     => $currency,
            'actual_price'        => $actual,
            'balance_price'       => $actual,
            'balance_currency'    => $currency,
            'exchange_rate'       => 0,
            'available_at'        => Carbon::now(),
            'status'              => Support::TRANSACTION_STATUS_PENDING,
            'source'              => $source,
            'type'                => $type,
            'outgoing_order_id'   => $outgoingOrderId,
            'extra'               => $extra,
        ];

        if ($holdingDays > 0) {
            $attributes = array_merge($attributes, [
                'available_at' => $attributes['available_at']->addDays($holdingDays),
            ]);
        }

        if ($balanceParams = $this->getRateService()->getBalancePrice($actual, $currency)) {
            $attributes = array_merge($attributes, $balanceParams);
        }

        /**
         * @var Transaction $model
         */
        $model = $this->getModel()->newInstance($attributes);

        $model->save();

        $model->refresh();

        /**
         * We assume transaction is pending, so we update statistic for pending case
         */
        $this->getStatisticRepository()->updateTransactionStatistic($owner, $model);

        /**
         * if no pending days, then we update to be approved
         */
        if (!$holdingDays) {
            $this->approveTransaction($model);
        }

        return $model;
    }

    public function createOutgoingTransaction(Order $order, ?string $gatewayOrderId = null): ?Transaction
    {
        if (!$order->user instanceof User) {
            return null;
        }

        if ($order->total < 0) {
            return null;
        }

        /**
         * TODO: Implement setting if need
         */
        $commissionPercentage = 0;

        if (null === $gatewayOrderId) {
            $gatewayOrderId = Payment::generateOrderId($order);
        }

        return $this->createModel($order->user, $order->user, $order->item, $order->currency, $order->total, $commissionPercentage, 0, Support::TRANSACTION_SOURCE_OUTGOING, Support::OUTGOING_TRANSACTION_TYPE_PURCHASED, $gatewayOrderId);
    }

    public function createIncomingTransaction(PaymentTransaction $transaction): ?Transaction
    {
        if ($transaction->status != PaymentTransaction::STATUS_COMPLETED) {
            return null;
        }

        /**
         * @var Order $order
         */
        $order = $transaction->order;

        if (null === $order) {
            return null;
        }

        if (!$order->payee instanceof User) {
            return null;
        }

        $total = $transaction->amount;

        if ($total <= 0) {
            return null;
        }

        return $this->createModel($transaction->user, $order->payee, $order->item, $transaction->currency, $total);
    }

    private function getStatisticRepository(): StatisticRepositoryInterface
    {
        return resolve(StatisticRepositoryInterface::class);
    }

    public function approveTransaction(Transaction $transaction): bool
    {
        $update = ['status' => Support::TRANSACTION_STATUS_APPROVED];

        $owner = $transaction->owner;

        if ($owner) {
            $userBalance = $this->getStatisticRepository()->getUserBalance($owner, $transaction->balance_currency);

            if (!is_numeric($userBalance)) {
                $userBalance = 0;
            }

            if ($transaction->source == Support::TRANSACTION_SOURCE_INCOMING) {
                $update['current_balance_price'] = $userBalance + $transaction->balance_price;
            } else {
                $update['current_balance_price'] = $userBalance > $transaction->balance_price ? $userBalance - $transaction->balance_price : 0;
            }
        }

        $transaction->update($update);

        $transaction->refresh();

        if ($owner) {
            $this->getStatisticRepository()->updateTransactionStatistic($owner, $transaction);

            /**
             * Only send approved notification when transaction is incoming from others who bought your items
             */
            if ($transaction->source == Support::TRANSACTION_SOURCE_INCOMING) {
                $this->sendApprovedNotification($owner, $transaction);
            }
        }

        return true;
    }

    protected function sendApprovedNotification(User $user, Transaction $transaction): void
    {
        $params = [$user, new ApprovedTransactionNotification($transaction)];

        Notification::send(...$params);
    }

    protected function getRateService(): ConversionRateServiceInterface
    {
        return resolve(ConversionRateServiceInterface::class);
    }

    public function viewTransactions(User $user, array $attributes = []): Paginator
    {
        $limit        = Arr::get($attributes, 'limit', Pagination::DEFAULT_ITEM_PER_PAGE);

        $query = $this->createBuilder($attributes);

        return $query
            ->where([
                'emoney_transactions.owner_id' => $user->entityId(),
            ])
            ->with(['userEntity', 'package'])
            ->orderByDesc('emoney_transactions.id')
            ->simplePaginate($limit, ['emoney_transactions.*']);
    }

    public function viewTransactionsAdminCP(array $attributes): Paginator
    {
        $limit        = Arr::get($attributes, 'limit', Pagination::DEFAULT_ITEM_PER_PAGE);

        $query = $this->createBuilder($attributes);

        return $query
            ->with(['userEntity', 'ownerEntity'])
            ->orderByDesc('emoney_transactions.id')
            ->paginate($limit, ['emoney_transactions.*']);
    }

    private function createBuilder(array $attributes): Builder
    {
        $baseCurrency = Arr::get($attributes, 'base_currency');
        $status       = Arr::get($attributes, 'status');
        $fromDate     = Arr::get($attributes, 'from_date');
        $toDate       = Arr::get($attributes, 'to_date');
        $buyer        = Arr::get($attributes, 'buyer');
        $seller       = Arr::get($attributes, 'seller');
        $id           = Arr::get($attributes, 'id');
        $source       = Arr::get($attributes, 'source');
        $type         = Arr::get($attributes, 'type');

        $query = $this->getModel()->newQuery();

        /**
         * For mobile old version that is not improving layout
         */
        if (!Emoney::isUsingNewAlias()) {
            $source = Support::TRANSACTION_SOURCE_INCOMING;
        }

        if ($source) {
            $query->where(['emoney_transactions.source' => $source]);
        }

        if ($type) {
            $query->where(['emoney_transactions.type' => $type]);
        }

        if (is_numeric($id)) {
            $query->where('emoney_transactions.id', $id);
        }

        if (is_string($buyer) && MetaFoxConstant::EMPTY_STRING != $buyer) {
            $query->join('user_entities', function (JoinClause $joinClause) use ($buyer) {
                $joinClause->on('user_entities.id', '=', 'emoney_transactions.user_id')
                    ->where('user_entities.name', $this->likeOperator(), '%' . $buyer . '%');
            });
        }

        if (is_string($seller) && MetaFoxConstant::EMPTY_STRING != $seller) {
            $query->join('user_entities as owner_entities', function (JoinClause $joinClause) use ($seller) {
                $joinClause->on('owner_entities.id', '=', 'emoney_transactions.owner_id')
                    ->where('owner_entities.name', $this->likeOperator(), '%' . $seller . '%');
            });
        }

        if ($baseCurrency) {
            $query->where('emoney_transactions.total_currency', $baseCurrency);
        }

        $scope = new GeneralScope($fromDate, $toDate, $status);

        $query->addScope($scope);

        return $query;
    }
}
