<?php

namespace MetaFox\EMoney\Http\Resources\v1\WithdrawRequest;

use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Support\Arr;
use MetaFox\EMoney\Facades\Emoney;
use MetaFox\EMoney\Policies\WithdrawRequestPolicy;
use MetaFox\EMoney\Repositories\StatisticRepositoryInterface;
use MetaFox\EMoney\Services\Contracts\WithdrawServiceInterface;
use MetaFox\EMoney\Support\Support;
use MetaFox\Form\AbstractForm;
use MetaFox\EMoney\Models\WithdrawRequest as Model;
use MetaFox\Form\Mobile\Builder;
use MetaFox\Platform\MetaFoxConstant;
use MetaFox\Yup\Yup;

/**
 * --------------------------------------------------------------------------
 * Form Configuration
 * --------------------------------------------------------------------------
 * stub: /packages/resources/edit_form.stub.
 */

/**
 * Class CreateWithdrawRequestForm.
 * @property ?Model $resource
 * @ignore
 * @codeCoverageIgnore
 */
class CreateWithdrawRequestMobileForm extends AbstractForm
{
    private string $currency = Support::DEFAULT_TARGET_CURRENCY_CONVERSION_RATE;

    protected function prepare(): void
    {
        $this->title(__p('ewallet::phrase.withdrawal_request'))
            ->action('emoney/request')
            ->setValue([
                'amount' => $this->getPriceValues(),
            ])
            ->asPost();
    }

    protected function initialize(): void
    {
        $context = user();

        $description = null;

        if (!policy_check(WithdrawRequestPolicy::class, 'validateMethod', $context, $this->currency)) {
            $description = __p('ewallet::phrase.there_are_no_withdrawal_providers_available');
        }

        if (!policy_check(WithdrawRequestPolicy::class, 'validateAmount', $context, $this->currency)) {
            $description = __p('ewallet::phrase.your_balance_is_not_enough_to_create_withdrawal_request_minimum_amount_is_number', ['number' => $this->getMinAmountFormatted()]);
        }

        if (null !== $description) {
            $this->addHeader(['showRightHeader' => false])
                ->component('FormHeader');

            $this->addBasic()
                ->addField(
                    Builder::typography()
                        ->label($description),
                );

            return;
        }

        $max = resolve(StatisticRepositoryInterface::class)->getUserBalance($context, $this->currency);
        $min = $this->getMinAmount();

        $minFormatted = $this->getMinAmountFormatted();

        $maxFormatted = app('currency')->getPriceFormatByCurrencyId($this->currency, $max);

        $this->addBasic()
            ->addFields(
                Builder::price('amount')
                    ->label(__p('ewallet::phrase.amount'))
                    ->description(__p('ewallet::admin.minimum_for_withdrawal_amount_is_number', ['number' => $minFormatted]))
                    ->required()
                    ->startAdornment($this->currency)
                    ->findReplace([
                        'find'    => [','],
                        'replace' => '.',
                    ])
                    ->yup(
                        Yup::array()
                            ->of(
                                Yup::object()
                                    ->addProperty(
                                        'value',
                                        Yup::number()
                                            ->required(__p('validation.field_is_a_required_field', [
                                                'field' => __p('ewallet::phrase.amount'),
                                            ]))
                                            ->min($min, __p('ewallet::validation.min_withdraw_value', ['number' => $minFormatted]))
                                            ->max($max, __p('ewallet::validation.max_withdraw_value', ['number' => $maxFormatted]))
                                            ->setError('typeError', __p('ewallet::admin.minimum_withdraw_format_is_invalid'))
                                    )
                            )
                    ),
                Builder::choice('withdraw_service')
                    ->required()
                    ->label(__p('ewallet::phrase.withdraw_via'))
                    ->enableSearch(false)
                    ->options($this->getMethodOptions())
                    ->enableSearch(false)
                    ->yup(
                        Yup::string()
                            ->required(__p('ewallet::validation.withdrawal_method_is_a_required_field'))
                    ),
            );
    }

    private function getMethodOptions(): array
    {
        return array_values(resolve(WithdrawServiceInterface::class)->getAvailableMethodsForUser(user()));
    }

    public function boot(): void
    {
        $context = user();

        if ($context->isGuest()) {
            throw new AuthorizationException();
        }
    }

    private function getMinAmount(): float
    {
        return Emoney::getMinimumWithdrawalAmount();
    }

    private function getMinAmountFormatted(): string
    {
        return app('currency')->getPriceFormatByCurrencyId($this->currency, $this->getMinAmount());
    }

    protected function getPriceValues(): array
    {
        $currencies      = app('currency')->getActiveOptions();
        $allowCurrencies = [Support::DEFAULT_TARGET_CURRENCY_CONVERSION_RATE];
        $name            = 'price_';

        return collect($currencies)
            ->filter(function ($currency) use ($allowCurrencies) {
                return in_array($currency['value'], $allowCurrencies);
            })
            ->map(function ($currency) use ($name) {
                return [
                    'name'  => $name . $currency['value'],
                    'label' => $currency['value'],
                    'value' => MetaFoxConstant::EMPTY_STRING,
                ];
            })
            ->values()
            ->all();
    }
}
