<?php

namespace MetaFox\EMoney\Http\Resources\v1\Transaction;

use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Carbon;
use MetaFox\EMoney\Facades\Emoney;
use MetaFox\EMoney\Models\Transaction as Model;
use MetaFox\User\Http\Resources\v1\UserEntity\UserEntityItem;
use MetaFox\User\Models\UserEntity;

/*
|--------------------------------------------------------------------------
| Resource Pattern
|--------------------------------------------------------------------------
| stub: /packages/resources/item.stub
*/

/**
 * Class TransactionItem.
 * @property Model $resource
 * @SuppressWarnings(PHPMD.UnusedFormalParameter)
 * @ignore
 * @codeCoverageIgnore
 * @mixin Model
 */
class TransactionItem extends JsonResource
{
    /**
     * Transform the resource collection into an array.
     *
     * @param  \Illuminate\Http\Request $request
     * @return array<string, mixed>
     */
    public function toArray($request)
    {
        $buyer = null;

        if ($this->resource->userEntity instanceof UserEntity) {
            $buyer = new UserEntityItem($this->resource->userEntity);
        }

        $currentBalance = null;

        if (is_numeric($this->resource->current_balance_price)) {
            $currentBalance = $this->getPriceFormat($this->resource->balance_currency, $this->resource->current_balance_price);
        }

        return [
            'id'            => $this->resource->entityId(),
            'module_name'   => Emoney::getAppAlias(),
            'resource_name' => $this->getResourceName(),
            'gross'         => $this->getPriceFormat($this->resource->total_currency, $this->resource->total_price),
            'fee'           => $this->getPriceFormat($this->resource->commission_currency, $this->resource->commission_price),
            'net'           => $this->getPriceFormat($this->resource->actual_currency, $this->resource->actual_price),
            'balance'       => $this->getPriceFormat($this->resource->balance_currency, $this->resource->balance_price),
            'current_balance' => $currentBalance,
            'status'        => $this->resource->status_text,
            'buyer'         => $buyer,
            'creation_date' => Carbon::parse($this->resource->created_at)->toISOString(),
            'source'        => $this->resource->source_text,
            'type'          => $this->resource->type_text,
            'reference'     => $this->getSource(),
        ];
    }

    protected function getResourceName(): string
    {
        if (Emoney::isUsingNewAlias()) {
            return $this->resource->entityType();
        }

        return 'emoney_transaction';
    }

    protected function getSource(): string
    {
        $source = $this->resource->package?->title;

        if (null === $source) {
            $source = __p('ewallet::phrase.unknown');
        }

        return $source;
    }

    protected function getPriceFormat(string $currency, float $price): string
    {
        return app('currency')->getPriceFormatByCurrencyId($currency, $price);
    }
}
