<?php

namespace MetaFox\EMoney\Http\Resources\v1\Statistic;

use Illuminate\Http\Resources\Json\JsonResource;
use MetaFox\EMoney\Facades\Emoney;
use MetaFox\EMoney\Models\Statistic as Model;
use MetaFox\EMoney\Repositories\StatisticRepositoryInterface;
use MetaFox\EMoney\Services\Contracts\ConversionRateServiceInterface;

/*
|--------------------------------------------------------------------------
| Resource Pattern
|--------------------------------------------------------------------------
| stub: /packages/resources/detail.stub
*/

/**
 * Class StatisticDetail.
 * @property Model $resource
 * @SuppressWarnings(PHPMD.UnusedFormalParameter)
 * @mixin Model
 */
class StatisticDetail extends JsonResource
{
    /**
     * Transform the resource collection into an array.
     *
     * @param  \Illuminate\Http\Request $request
     * @return array<string, mixed>
     */
    public function toArray($request)
    {
        $user = $this->resource->user;

        if (null === $user) {
            return [];
        }

        $userCurrency = app('currency')->getUserCurrencyId($user);

        $balanceCurrency = $this->resource->currency;

        $conversionRate = null;

        if ($userCurrency != $balanceCurrency) {
            $conversionRate = app('ewallet.conversion-rate')->getExchangeRate($userCurrency, $balanceCurrency);
        }

        return [
            'id'                  => $this->resource->entityId(),
            'module_name'         => Emoney::getAppAlias(),
            'resource_name'       => $this->getResourceName(),
            'earned'              => $this->getAmountFormat(...$this->getEarnedAmount($userCurrency)),
            'balances'            => resolve(StatisticRepositoryInterface::class)->getUserBalances($user),
            'user_balance'        => $this->getAmountFormat(...$this->getUserBalanceAmount($userCurrency)),
            'pending_withdraw'    => $this->getAmountFormat($this->resource->currency, $this->resource->total_pending),
            'pending_transaction' => $this->getAmountFormat($this->resource->currency, $this->resource->total_pending_transaction),
            'withdraw'            => $this->getAmountFormat(...$this->getWithdrawnAmount($userCurrency)),
            'purchased'           => $this->getAmountFormat(...$this->getPurchasedAmount($userCurrency)),
            'balance_description' => __p('ewallet::phrase.balance_description', [
                'hasConversionRate' => is_numeric($conversionRate) ? 1 : 0,
                'baseCurrency'      => $userCurrency,
                'targetPrice'       => $conversionRate,
                'targetCurrency'    => $balanceCurrency,
            ])
        ];
    }

    private function getResourceName(): string
    {
        if (Emoney::isUsingNewAlias()) {
            return $this->resource->entityType();
        }

        return 'emoney_statistic';
    }

    protected function getPurchasedAmount(string $userCurrency): array
    {
        $purchased = $this->getConversedAmount($this->resource->currency, $this->resource->total_purchased, $userCurrency);

        if (null === $purchased) {
            return [$this->resource->currency, $this->resource->total_withdrawn];
        }

        return [$userCurrency, $purchased];
    }

    protected function getWithdrawnAmount(string $userCurrency): array
    {
        $withdrawn = $this->getConversedAmount($this->resource->currency, $this->resource->total_withdrawn, $userCurrency);

        if (null === $withdrawn) {
            return [$this->resource->currency, $this->resource->total_withdrawn];
        }

        return [$userCurrency, $withdrawn];
    }

    protected function getEarnedAmount(string $userCurrency): array
    {
        $earned = $this->getConversedAmount($this->resource->currency, $this->resource->total_earned, $userCurrency);

        if (null === $earned) {
            return [$this->resource->currency, $this->resource->total_earned];
        }

        return [$userCurrency, $earned];
    }

    protected function getUserBalanceAmount(string $userCurrency): array
    {
        $balance = $this->getConversedAmount($this->resource->currency, $this->resource->total_balance, $userCurrency);

        if (null === $balance) {
            return [$this->resource->currency, $this->resource->total_balance];
        }

        return [$userCurrency, $balance];
    }

    protected function getConversedAmount(string $baseCurrency, float $baseAmount, string $targetCurrency): ?float
    {
        return resolve(ConversionRateServiceInterface::class)->getConversedAmount($baseCurrency, $baseAmount, $targetCurrency);
    }

    protected function getAmountFormat(string $currency, float $amount): string
    {
        return app('currency')->getPriceFormatByCurrencyId($currency, $amount);
    }
}
