<?php

namespace MetaFox\EMoney\Http\Requests\v1\WithdrawRequest;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Arr;
use MetaFox\EMoney\Facades\Emoney;
use MetaFox\EMoney\Repositories\StatisticRepositoryInterface;
use MetaFox\EMoney\Services\Contracts\WithdrawServiceInterface;
use MetaFox\Platform\MetaFox;
use MetaFox\Platform\Rules\AllowInRule;

/**
 * --------------------------------------------------------------------------
 *  Http request for api version v1
 * --------------------------------------------------------------------------.
 *
 * This class is used by automatic dependency injection:
 *
 * @link \MetaFox\EMoney\Http\Controllers\Api\v1\WithdrawRequestController::store
 * stub: /packages/requests/api_action_request.stub
 */

/**
 * Class StoreRequest.
 */
class StoreRequest extends FormRequest
{
    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, mixed>
     */
    public function rules()
    {
        $context = user();
        $methods = array_column(resolve(WithdrawServiceInterface::class)->getAvailableMethodsForUser($context), 'value');

        $min = Emoney::getMinimumWithdrawalAmount();
        $max = resolve(StatisticRepositoryInterface::class)->getUserBalance($context, Emoney::getDefaultCurrency());

        $rules = [
            'amount'           => ['required', 'numeric', 'min: ' . $min, 'max: ' . $max],
            'withdraw_service' => ['required', new AllowInRule($methods)],
        ];

        return $this->handlePriceRule($rules, $max, $min);
    }

    private function handlePriceRule(array $rules, int $max, int $min): array
    {
        if (!MetaFox::isMobile()) {
            return $rules;
        }

        $rules['amount']         = ['required', 'array', 'min:1', 'max:1'];
        $rules['amount.*.value'] = ['required', 'numeric', 'min: ' . $min, 'max: ' . $max];

        return $rules;
    }

    public function validated($key = null, $default = null)
    {
        $data = parent::validated($key, $default);

        return array_merge($this->handleTransformData($data), [
            'currency' => Emoney::getDefaultCurrency(),
        ]);
    }

    private function handleTransformData(array $data): array
    {
        if (!MetaFox::isMobile()) {
            return $data;
        }

        $amounts     = Arr::get($data, 'amount', []);
        $amountValue = Arr::first($amounts)['value'] ?? 0;

        $data['amount'] = $amountValue;

        return $data;
    }

    public function messages()
    {
        $context = user();

        $currency = Emoney::getDefaultCurrency();

        $max = resolve(StatisticRepositoryInterface::class)->getUserBalance($context, $currency);

        $min = app('currency')->getPriceFormatByCurrencyId($currency, Emoney::getMinimumWithdrawalAmount());

        if (is_numeric($max)) {
            $max = app('currency')->getPriceFormatByCurrencyId($currency, $max);
        }

        $messages = [
            'amount.required' => __p('validation.field_is_a_required_field', [
                'field' => __p('ewallet::phrase.amount'),
            ]),
            'amount.min'                => __p('ewallet::validation.min_withdraw_value', ['number' => $min]),
            'amount.max'                => __p('ewallet::validation.max_withdraw_value', ['number' => $max]),
            'withdraw_service.required' => __p('ewallet::validation.withdrawal_method_is_a_required_field'),
        ];

        return $this->handleMessage($messages, $max, $min);
    }

    private function handleMessage(array $messages, ?string $max, ?string $min): array
    {
        if (!MetaFox::isMobile()) {
            return $messages;
        }

        $messages['amount.*.value.min'] = __p('ewallet::validation.min_withdraw_value', ['number' => $min]);
        $messages['amount.*.value.max'] = __p('ewallet::validation.max_withdraw_value', ['number' => $max]);

        return $messages;
    }
}
