<?php
/**
 * @author  developer@phpfox.com
 * @license phpfox.com
 */

namespace MetaFox\Core\Listeners;

use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use MetaFox\Core\Repositories\DriverRepositoryInterface;
use MetaFox\Platform\Facades\Settings;
use MetaFox\Platform\PackageManager;
use MetaFox\Platform\Support\BasePackageSettingListener;

/**
 * Handle module installed.
 *
 * Class PackageInstalledListener
 */
class PackageInstalledListener
{
    public function handle(string $package): void
    {
        $this->publishSiteSettings($package);
        $this->publishDrivers($package);
        $this->publishSiteStats($package);
    }

    /**
     * publish site settings into the database.
     *
     * @param string $package
     */
    private function publishSiteSettings(string $package): void
    {
        Log::channel('installation')->debug('publishSiteSettings', [$package]);

        /** @var null|BasePackageSettingListener $listener */
        $listener = PackageManager::getListener($package);

        if (!$listener) {
            return;
        }

        $moduleId = PackageManager::getAlias($package);

        $settings = $listener->getSiteSettings();

        //        Log::channel('installation')->debug('setupPackageSettings', $settings);

        Settings::setupPackageSettings($moduleId, $settings);
    }

    /**
     * Import drivers from "resources/drivers.php".
     *
     * @param string $package
     */
    private function publishDrivers(string $package): void
    {
        Log::channel('installation')->debug('publishDrivers', [$package]);

        $drivers = PackageManager::readFile($package, 'resources/drivers.php');

        if ($drivers) {
            resolve(DriverRepositoryInterface::class)->setupDrivers($package, $drivers);
        }
    }

    private function publishSiteStats(string $package): void
    {
        $data   = PackageManager::discoverSettings('getSiteStatContent');
        $icons  = [];
        $urls   = [];
        foreach ($data as $config) {
            if (!is_array($config)) {
                continue;
            }

            foreach ($config as $itemType => $stat) {
                // Old version compatible
                if (is_string($stat)) {
                    // $stat is icon
                    $icons[$itemType] = $stat;
                    continue;
                }

                if (Arr::has($stat, 'to')) {
                    Arr::set($urls, "$itemType", $stat['to']);
                }

                $icon = Arr::get($stat, 'icon') ?? 'ico-square-o';
                Arr::set($icons, $itemType, $icon);
            }
        }

        Settings::save(['core.general.site_stat_icons' => $icons]);
        Settings::save(['core.general.site_stat_urls' => $urls]);
    }
}
