<?php

namespace MetaFox\Activity\Observers;

use MetaFox\Activity\Models\Feed;
use MetaFox\Activity\Repositories\FeedRepositoryInterface;
use MetaFox\Activity\Repositories\PinRepositoryInterface;
use MetaFox\Activity\Support\Facades\ActivityFeed;
use MetaFox\Platform\Contracts\ActivityFeedSource;
use MetaFox\Platform\Contracts\Content;
use MetaFox\Platform\MetaFoxConstant;

/**
 * Class FeedObserver.
 */
class FeedObserver
{
    public function __construct(protected FeedRepositoryInterface $repository)
    {
    }
    /**
     * Handle the ActivityFeed "created" event.
     *
     * @param Feed $model
     */
    public function created(Feed $model): void
    {
        ActivityFeed::putToStream($model);

        app('events')->dispatch('search.created', [$model], true);
    }

    /**
     * Handle the ActivityFeed "updated" event.
     *
     * @param Feed $model
     */
    public function updated(Feed $model): void
    {
        $model->stream()->delete();

        ActivityFeed::putToStream($model);
        app('events')->dispatch('search.updated', [$model], true);

        if ($model->status == MetaFoxConstant::ITEM_STATUS_REMOVED) {
            $this->repository->handleRemoveNotification('activity_feed_approved', $model->entityId(), $model->entityType());
        }
    }

    /**
     * Handle the ActivityFeed "deleted" event.
     *
     * @param Feed $model
     */
    public function deleted(Feed $model): void
    {
        $model->stream()->delete();

        $model->tagData()->sync([]);

        app('events')->dispatch('search.deleted', [$model], true);

        if ($model->item instanceof ActivityFeedSource) {
            if ($model->from_resource == Feed::FROM_FEED_RESOURCE) {
                app('events')->dispatch('activity.feed.deleted', [$model->item]);
            }
        }

        //Delete notification types: activity_feed_approved, activity_feed_declined
        app('events')->dispatch(
            'notification.delete_mass_notification_by_item',
            [$model],
            true
        );

        $model->history()->delete();

        $model->pinned()->delete();

        resolve(PinRepositoryInterface::class)->clearCache();

        $this->disableItemSponsorInFeed($model);
    }

    protected function disableItemSponsorInFeed(Feed $model): void
    {
        if (!$model->is_sponsor) {
            return;
        }

        app('events')->dispatch('advertise.sponsor.delete_by_item', [$model]);

        /*
         * We let feed controlling disable sponsor_in_feed of Item
         */
        if ($model->relationLoaded('item')) {
            $model->load('item');
        }

        if (!$model->item instanceof Content) {
            return;
        }

        app('events')->dispatch('advertise.sponsor.disable_sponsor_feed', [$model->item]);
    }
}
