<?php

namespace MetaFox\Mfa\Http\Controllers\Api\v1;

use Illuminate\Http\JsonResponse;
use Illuminate\Support\Arr;
use MetaFox\Mfa\Http\Requests\v1\UserAuth\AuthRequest;
use MetaFox\Mfa\Http\Requests\v1\UserAuth\FormRequest;
use MetaFox\Mfa\Http\Requests\v1\UserAuth\RemoveAuthenticationRequest;
use MetaFox\Mfa\Http\Requests\v1\UserAuth\ResendVerificationAuthRequest;
use MetaFox\Platform\Http\Controllers\Api\ApiController;
use MetaFox\Mfa\Repositories\UserServiceRepositoryInterface;
use MetaFox\Mfa\Support\Facades\Mfa;
use MetaFox\User\Http\Resources\v1\User\Admin\UserItem;
use MetaFox\User\Models\User;
use Prettus\Validator\Exceptions\ValidatorException;

/**
 * Class UserAuthController.
 * @codeCoverageIgnore
 * @SuppressWarnings(PHPMD.UnusedLocalVariable)
 * @ignore
 */
class UserAuthController extends ApiController
{
    /**
     * @var UserServiceRepositoryInterface
     */
    private UserServiceRepositoryInterface $repository;

    /**
     * UserServiceController Constructor.
     *
     * @param UserServiceRepositoryInterface $repository
     */
    public function __construct(UserServiceRepositoryInterface $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Setup service form.
     *
     * @param  FormRequest        $request
     * @return JsonResponse
     * @throws ValidatorException
     */
    public function form(FormRequest $request): JsonResponse
    {
        $params     = $request->validated();
        $mfaToken   = Arr::get($params, 'mfa_token', '');
        $service    = Arr::get($params, 'service');
        $resolution = Arr::get($params, 'resolution', 'web');

        $form = Mfa::loadServiceSelectionForm($mfaToken, $resolution);

        if (null != $service) {
            $form = Mfa::loadAuthForm($mfaToken, $service, $resolution);
        }

        return $this->success($form, $form->getMultiStepFormMeta());
    }

    /**
     * Auth user.
     *
     * @param  AuthRequest        $request
     * @throws ValidatorException
     */
    public function auth(AuthRequest $request)
    {
        $response = Mfa::authenticate($request);

        return is_array($response) ? $this->success($response) : $response;
    }

    public function resendVerificationAuth(ResendVerificationAuthRequest $request): JsonResponse
    {
        Mfa::resendVerificationAuth($request);

        return $this->success([], [], __p('mfa::phrase.verification_code_successfully'));
    }

    public function removeAuthentication(RemoveAuthenticationRequest $request): JsonResponse
    {
        $params = $request->validated();

        /** @var ?User $owner */
        $owner = User::query()->findOrFail($params['user_id']);

        Mfa::deactivate($owner, $params['service']);

        return $this->success(new UserItem($owner), [], __p('user::phrase.user_removed_authentication_successfully'));
    }
}
