<?php

namespace MetaFox\Localize\Form\Html;

use MetaFox\Core\Support\Facades\Language;
use MetaFox\Form\AbstractField;
use MetaFox\Form\Builder;
use MetaFox\Form\Section;
use MetaFox\Yup\Shape;
use MetaFox\Yup\Yup;

class TranslatableTextField extends Section
{
    private string $label = '';

    private ?AbstractField $defaultComponent = null;

    private ?string $componentText;

    private ?Shape $validator;

    public function __construct(array $properties = [])
    {
        parent::__construct($properties);

        $this->componentText = 'text';
        $this->validator     = null;
    }

    public function initialize(): void
    {
        parent::initialize();

        $this->sx(['mb' => 2]);
    }

    /**
     * NOTE: This method must be use lastly in order to other setAttribute to taking affect.
     */
    public function buildFields(): self
    {
        $default                = Language::getDefaultLocaleId();
        $defaultLanguage        = Language::getLanguage($default);
        $name                   = $this->getName();
        $this->defaultComponent = $this->getComponent(sprintf('%s.%s', $name, $defaultLanguage->language_code));

        $this->addField(
            $this->defaultComponent()
                ->required((bool) $this->getAttribute('required'))
                ->fullWidth(false)
                ->marginDense()
                ->label(__p('localize::phrase.name_in_language_name', ['name' => $this->label, 'language' => $defaultLanguage->name]))
                ->description(__p('localize::phrase.translatable_text_desc'))
                ->yup($this->getYup()),
        );

        $languages = Language::getAllActiveLanguages();
        if (count($languages) == 1) {
            return $this;
        }

        foreach ($languages as $locale => $language) {
            if ($locale === $default) {
                continue;
            }

            $this->addField(
                $this->getComponent(sprintf('%s.%s', $name, $language->language_code))
                    ->fullWidth(false)
                    ->marginDense()
                    ->label(__p('localize::phrase.name_in_language_name', ['name' => $this->label, 'language' => $language->name]))
                    ->showWhen(['truthy', sprintf('%s.%s', $name, 'use_custom_language')])
            );
        }

        $this->addField(
            Builder::viewMore(sprintf('%s.%s', $name, 'use_custom_language'))
                ->marginDense()
                ->viewMoreText(__p('localize::phrase.name_in_other_language', ['name' => $this->label]))
                ->viewLessText(__p('localize::phrase.use_default'))
        );

        return $this;
    }

    public function label(?string $label): static
    {
        $this->label = $label;

        return $this;
    }

    /**
     * get validation.
     *
     * @return Shape
     */
    public function getYup(): Shape
    {
        $validator = $this->validator;
        if (null != $validator) {
            return $validator;
        }

        $validator = Yup::string();
        if ($this->getAttribute('required', false)) {
            $validator->required();
        }

        return $validator;
    }

    public function yup(Shape $validator): static
    {
        $this->validator = $validator;

        return $this;
    }

    protected function getComponent(string $name): AbstractField
    {
        $creator = Builder::getCreator($this->componentText);

        return new $creator(['name' => $name]);
    }

    public function asTextArea(): self
    {
        $this->componentText = 'textArea';

        return $this;
    }

    public function asTextEditor(): self
    {
        $this->componentText = 'richTextEditor';

        return $this;
    }

    public function defaultComponent(): ?AbstractField
    {
        return $this->defaultComponent;
    }
}
