<?php

namespace MetaFox\HealthCheck\Checks;

use Illuminate\Support\Carbon;
use MetaFox\Platform\HealthCheck\Checker;
use MetaFox\Platform\HealthCheck\Result;

class CheckQueueWorker extends Checker
{
    protected ?string $cacheKey = 'health:checks:queue:latestHeartbeatAt';

    protected ?string $cacheStoreName = null;

    protected int $failWhenTestJobTakesLongerThanMinutes = 5;

    public function getCacheKey(?string $queue): string
    {
        $queue = $queue ?? 'default';

        return "{$this->cacheKey}.{$queue}";
    }

    public function check(): Result
    {
        $queues = ['default'];
        $result = $this->makeResult();

        $result->debug(__p('queue::phrase.default_label') . ': ' . config('queue.default', 'database'));

        foreach ($queues as $queue) {
            $lastHeartbeatTimestamp = cache()->get($this->getCacheKey($queue));

            if (!$lastHeartbeatTimestamp) {
                $result->error(__p('health-check::phrase.the_queue_did_not_run_yet', ['queue' => $queue]));
                continue;
            }

            $latestHeartbeatAt = Carbon::createFromTimestamp($lastHeartbeatTimestamp);

            $minutesAgo = $latestHeartbeatAt->diffInMinutes() + 1;

            if ($minutesAgo > $this->failWhenTestJobTakesLongerThanMinutes) {
                $result->error(__p('health-check::phrase.the_last_run_of_the_queue_was_more_than_minutes_ago', [
                    'queue'      => $queue,
                    'minutesAgo' => $minutesAgo,
                ]));
                continue;
            }

            // pass
            $result->success(__p('health-check::phrase.the_last_run_of_the_queue_was_more_than_minutes_ago', [
                'queue'      => $queue,
                'minutesAgo' => $minutesAgo,
            ]));
        }

        return $result;
    }

    public function getName()
    {
        return __p('health-check::phrase.queues');
    }
}
