<?php

namespace MetaFox\App\Http\Controllers\Api\v1;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use MetaFox\App\Http\Requests\v1\AppStoreProduct\Admin\IndexRequest;
use MetaFox\App\Http\Requests\v1\AppStoreProduct\Admin\SearchFormRequest;
use MetaFox\App\Http\Resources\v1\AppStoreProduct\Admin\SearchForm;
use MetaFox\App\Http\Resources\v1\AppStoreProduct\Admin\StoreAppDetail;
use MetaFox\App\Repositories\PackageRepositoryInterface;
use MetaFox\App\Support\MetaFoxStore;
use MetaFox\App\Support\PackageInstaller;
use MetaFox\Platform\Http\Controllers\Api\ApiController;

class StoreAdminController extends ApiController
{
    /**
     * @var MetaFoxStore
     */
    private MetaFoxStore $store;

    private PackageRepositoryInterface $packageRepository;

    public function __construct(PackageRepositoryInterface $packageRepository)
    {
        $this->packageRepository = $packageRepository;

        $this->store = resolve(MetaFoxStore::class);
    }

    public function index(IndexRequest $request): JsonResponse
    {
        $params = $request->validated();

        $products = Cache::remember(
            'STORE_' . md5(implode('', $params)),
            10,
            function () use ($params) {
                return $this->store->browse($params);
            }
        );

        return $this->success($products);
    }

    public function show(int $id): JsonResponse
    {
        $data = $this->store->show($id);

        return $this->success(new StoreAppDetail($data));
    }

    /**
     * View search form.
     *
     * @param  SearchFormRequest $request
     * @return JsonResponse
     */
    public function form(SearchFormRequest $request): JsonResponse
    {
        $data = $this->store->getSearchFormData();

        return $this->success(new SearchForm($data));
    }

    public function install(Request $request): JsonResponse
    {
        $storeAppId      = $request->get('app_id');
        $name            = $request->get('name');
        $app_version     = $request->get('app_version');
        $release_channel = $request->get('release_channel');

        try {
            $this->packageRepository->setInstallationStatus($name, 'downloading');

            $filename = $this->store->downloadProduct($name, $app_version, $release_channel);

            $this->packageRepository->setInstallationStatus($name, 'installing');

            resolve(PackageInstaller::class)->install($filename);

            $this->packageRepository->setInstallationStatus($name, '');

            Artisan::call('optimize:clear');
        } catch (Exception $exception) {
            Log::channel('installation')->debug('\MetaFox\App\Http\Controllers\Api\v1\StoreAdminController::install failed ' . $exception->getMessage() . $exception->getTraceAsString());

            $this->packageRepository->setInstallationStatus($name, '');

            return $this->error($exception->getMessage(), 402);
        }

        $data = $this->store->show($storeAppId);

        return $this->success($data, [], __p('core::phrase.installed_successfully'));
    }
}
